use super::context as ctx;
use crate::config::Config;
use chrono::{DateTime, Local};
use std::borrow::Cow;
use std::path::Path;
use std::time::SystemTime;
use std::{fs, io};

pub struct EntryDataBuilder<'a> {
	pub dir_user_path: &'a str,
	pub entry: fs::DirEntry,
	pub config: &'a Config,
}

impl EntryDataBuilder<'_> {
	pub fn try_build(self) -> io::Result<ctx::EntryData> {
		let metadata = self.entry.metadata()?;
		let file_type = metadata.file_type();
		let path = self.entry.path();
		let size = if file_type.is_dir() {
			use crate::config::DirectorySizeOption as D;
			match self.config.directory_size {
				D::NoSize => None,
				D::Naive => Some((metadata.len(), true)),
				D::Entries => Some((num_dir_entries(&path, self.config.exclude_dotfiles)?, false)),
				D::Recursive => Some((dir_size_recursive(&path, self.config.exclude_dotfiles)?, true)),
			}
		} else {
			Some((metadata.len(), true))
		};
		Ok(ctx::EntryData {
			name: self.entry.file_name().to_string_lossy().to_string(),
			thumbnail: Self::make_thumbnail(file_type, Cow::Borrowed(&path), (self.dir_user_path, &self.entry.file_name().to_string_lossy()))?,
			modified_time: Self::make_last_modified(metadata.modified()?),
			size: size.map(Self::make_size),
		})
	}
	fn make_last_modified(time: SystemTime) -> ctx::Friendly<i64> {
		// XXX try DateTime<Utc> too
		let time = DateTime::<Local>::from(time);
		ctx::Friendly {
			number: time.timestamp(),
			friendly: time.format("%Y-%m-%d %H:%M:%S").to_string(),
		}
	}
	fn make_size((size, should_format): (u64, bool)) -> ctx::Friendly<u64> {
		use humansize::{file_size_opts as opts, FileSize as _};
		ctx::Friendly {
			number: size,
			friendly: if should_format { size.file_size(opts::BINARY).unwrap() } else { size.to_string() },
		}
	}
	fn thumbnail_path((parent, basename): (&str, &str)) -> String {
		let parent = parent.trim_start_matches('/');
		if parent.is_empty() {
			format!("/thumbnail/64/{}", basename)
		} else {
			format!("/thumbnail/64/{}/{}", parent, basename)
		}
	}
	fn make_thumbnail(mut file_type: fs::FileType, mut path: Cow<'_, Path>, user_path_parts: (&str, &str)) -> io::Result<ctx::ThumbnailData> {
		use ctx::{ThumbnailData, ThumbnailType};

		while file_type.is_symlink() {
			path = crate::util::dereference_symlink(&path)?.into();
			file_type = path.metadata()?.file_type();
		}

		Ok(if file_type.is_file() {
			use crate::util::MimeIsRich as _;
			let mime = crate::util::get_file_mime(&path)?;
			if let Some(rich_type) = mime.richness() {
				ThumbnailData {
					alt: ThumbnailType::RichThumbnail(rich_type),
					url: Self::thumbnail_path(user_path_parts).into(),
				}
			} else {
				ThumbnailData {
					alt: ThumbnailType::File,
					url: "/static/img/file.svg".into(),
				}
			}
		} else if file_type.is_dir() {
			ThumbnailData {
				alt: ThumbnailType::Directory,
				url: "/static/img/directory.svg".into(),
			}
		} else {
			ThumbnailData {
				alt: ThumbnailType::Unknown,
				url: "/static/img/unknown.svg".into(),
			}
		})
	}
}

fn num_dir_entries(path: &Path, exclude_dotfiles: bool) -> io::Result<u64> {
	let mut count = 0;
	for entry in fs::read_dir(path)? {
		let entry = entry?;
		if filter_entry(&entry, exclude_dotfiles) {
			count += 1;
		}
	}
	Ok(count.try_into().unwrap())
}

fn dir_size_recursive(path: &Path, exclude_dotfiles: bool) -> io::Result<u64> {
	let mut size = 0;
	for entry in fs::read_dir(path)? {
		let entry = entry?;
		if !filter_entry(&entry, exclude_dotfiles) {
			continue;
		}
		size += if entry.file_type()?.is_dir() { dir_size_recursive(&entry.path(), exclude_dotfiles)? } else { entry.metadata()?.len() };
	}
	Ok(size)
}

pub fn filter_entry(entry: &fs::DirEntry, exclude_dotfiles: bool) -> bool {
	match entry.file_name().to_str() {
		Some(s) => !exclude_dotfiles || !s.starts_with('.'),
		// if it's invalid Unicode, it will be refused later anyway
		None => false,
	}
}
