use image as im;
use rocket::http::Status;
use std::path::Path;
use std::{fs, io};
use vid2img as vi;

pub fn io_error_to_status(err: io::Error) -> Status {
	match err.kind() {
		io::ErrorKind::NotFound => Status::NotFound,
		_ => {
			log::error!("IO error: {:?}: {}", err.kind(), err);
			Status::InternalServerError
		}
	}
}

pub fn image_error_to_status(err: im::ImageError) -> Status {
	match err {
		im::ImageError::IoError(err) => io_error_to_status(err),
		_ => {
			log::error!("Image error: {}", err);
			Status::InternalServerError
		}
	}
}

pub fn vid2img_error_to_status(err: vi::CaptureError) -> Status {
	match err {
		vi::CaptureError::IoError(err) => io_error_to_status(err),
	}
}

pub fn stream_error_to_status(err: vi::StreamError) -> Status {
	match err {
		vi::StreamError::GstError(underlying) => {
			log::error!("GST stream error: {:?}", underlying);
			Status::InternalServerError
		}
		vi::StreamError::FrameCaptureError => Status::NotFound, // if a frame couldn't be captured, it's probably not there to be captured
	}
}

pub fn get_file_mime(path: &Path) -> io::Result<Option<mime::Mime>> {
	use mime_sniffer::MimeTypeSnifferExt;
	let file = fs::File::open(path)?;
	let data = match unsafe { memmap::Mmap::map(&file) } {
		// if the file is empty, the MMap will fail, but we wouldn't be able to determine the MIME anyway, so just return `None`.
		Err(err) if err.kind() == io::ErrorKind::InvalidInput => {
			return Ok(None);
		}
		x => x?,
	};
	Ok(data.sniff_mime_type_ext())
}

pub trait MimeIsRich {
	fn richness(&self) -> Option<RichType>;
}

impl MimeIsRich for Option<mime::Mime> {
	fn richness(&self) -> Option<RichType> {
		self.as_ref().and_then(MimeIsRich::richness)
	}
}

impl MimeIsRich for mime::Mime {
	fn richness(&self) -> Option<RichType> {
		let ty = self.type_().as_str();
		Some(match ty {
			"image" => RichType::Image,
			"video" => RichType::Video,
			_ => return None,
		})
	}
}

#[derive(serde::Serialize)]
#[serde(rename_all = "snake_case")]
pub enum RichType {
	Image,
	Video,
}
