use dessin::{
    shape::{EmbeddedDrawing, Line, Style},
    vec2, Drawing, Rect, Shape,
};

pub struct Rectangle {
    pub(crate) pos: Rect,
    pub(crate) style: Option<Style>,
}
dessin::impl_pos!(Rectangle);
dessin::impl_style!(Rectangle);
impl Rectangle {
    pub const fn new() -> Self {
        Rectangle {
            pos: Rect::new(),
            style: None,
        }
    }
}

impl Into<Shape> for Rectangle {
    fn into(self) -> Shape {
        let mut rect = Drawing::empty().with_canvas_size(vec2(2., 2.));
        rect.add(
            Line::from(vec2(-1., -1.))
                .to(vec2(-1., 1.))
                .with_style(self.style.as_ref().map(|v| v.clone()).unwrap_or_default()),
        )
        .add(
            Line::from(vec2(-1., -1.))
                .to(vec2(1., -1.))
                .with_style(self.style.as_ref().map(|v| v.clone()).unwrap_or_default()),
        )
        .add(
            Line::from(vec2(1., 1.))
                .to(vec2(-1., 1.))
                .with_style(self.style.as_ref().map(|v| v.clone()).unwrap_or_default()),
        )
        .add(
            Line::from(vec2(1., 1.))
                .to(vec2(1., -1.))
                .with_style(self.style.as_ref().map(|v| v.clone()).unwrap_or_default()),
        );

        EmbeddedDrawing::new(rect)
            .at(self.pos.position_from_center())
            .with_size(self.pos.size())
            .into()
    }
}

#[cfg(test)]
mod test_rectangle {
    use super::*;
    use dessin::ShapeType;

    #[test]
    fn rect() {
        let rect: Shape = Rectangle::new()
            .at(vec2(10., 10.))
            .with_size(vec2(20., 20.))
            .into();

        if let ShapeType::Drawing(shapes) = &rect.shape_type {
            if let ShapeType::Line { from, to } = &shapes[0].shape_type {
                assert_eq!(*from, vec2(0., 0.));
                assert_eq!(*to, vec2(0., 20.));
            } else {
                panic!("Expected a Line");
            }

            if let ShapeType::Line { from, to } = &shapes[1].shape_type {
                assert_eq!(*from, vec2(0., 0.));
                assert_eq!(*to, vec2(20., 0.));
            } else {
                panic!("Expected a Line");
            }

            if let ShapeType::Line { from, to } = &shapes[2].shape_type {
                assert_eq!(*from, vec2(20., 20.));
                assert_eq!(*to, vec2(0., 20.));
            } else {
                panic!("Expected a Line");
            }

            if let ShapeType::Line { from, to } = &shapes[3].shape_type {
                assert_eq!(*from, vec2(20., 20.));
                assert_eq!(*to, vec2(20., 0.));
            } else {
                panic!("Expected a Line");
            }
        } else {
            panic!("Expected a Drawing");
        }
    }
}
