use num_derive::FromPrimitive;

/// <summary>
/// hold the Access Rights for changing application keys (Change Key command)
/// </summary>
#[repr(u8)]
#[derive(Clone, Copy, FromPrimitive)]
pub enum ChangeApplicationKey {
    /// <summary>
    /// Application master key authentication is necessary to change any key (default)
    /// </summary>
    MASTERKEY = 0x00,
    /// <summary>
    /// Authentication with the key to be changed (same Key#) is necessary to change a key
    /// </summary>
    SAMEKEY = 0x0E,
    /// <summary>
    /// All keys (except application master key, see Bit 0) within this application are frozen
    /// </summary>
    ALLKEYS = 0x0F,
}

/// <summary>
/// codes whether the application master key is changeable
/// </summary>
#[repr(u8)]
#[derive(Clone, Copy)]
pub enum ChangeMasterKey {
    /// <summary>
    /// Application master key is not changeable anymore (frozen)
    /// </summary>
    FROZEN = 0x00,

    /// <summary>
    /// Application master key is changeable (authentication with the current application master key necessary, default)
    /// </summary>
    CHANGEABLE = 0x01,
}

/// <summary>
/// codes whether a change of the application master key settings is allowed
/// </summary>
#[repr(u8)]
#[derive(Clone, Copy)]
pub enum ChangeMasterKeySettings {
    /// <summary>
    /// configuration not changeable anymore (frozen)
    /// </summary>
    FROZEN = 0x00,

    /// <summary>
    /// this configuration is changeable if authenticated with the application master key (default)
    /// </summary>
    WITHMASTERKEY = 0x08,
}

/// <summary>
/// codes whether application master key authentication is needed before “Create File” / “Delete File”
/// </summary>
#[repr(u8)]
#[derive(Clone, Copy)]
pub enum CreateDeleteFile {
    /// <summary>
    /// “Create File”/ “Delete File”is permitted only with application master key authentication
    /// </summary>
    ONLYMASTERKEY = 0x00,

    /// <summary>
    /// “Create File”/ “Delete File”is permitted also without application master key authentication (default)
    /// </summary>
    NOKEY = 0x04,
}

/// <summary>
/// Crypto method of the application
/// </summary>
#[repr(u8)]
#[derive(Clone, Copy)]
pub enum CryptoOperationsType {
    TDES = 0x00,
    TKTDES = 0x40,
    AES = 0x80,
}

#[repr(u8)]
#[derive(Clone, Copy)]
pub enum FileAccessRights {
    FREE = 0x0E,
    NEVER = 0x0F,
}

#[repr(u8)]
#[derive(Clone, Copy)]
pub enum FileCommunication {
    /// <summary>
    /// "Plain communication"
    /// </summary>
    PLAIN = 0x00,

    /// <summary>
    /// Plain communication secured by DES/3DES MACing
    /// </summary>
    MAC = 0x01,

    /// <summary>
    /// Fully DES/3DES enciphered communication
    /// </summary>
    ENCRYPT = 0x03,
}

/// <summary>
/// codes whether application master key authentication is needed for file directory access
/// </summary>
#[repr(u8)]
#[derive(Clone, Copy)]
pub enum FileDirectoryAccess {
    /// <summary>
    /// Successful application master key authentication is required for executing the “Get FID List”, “Get File Settings”and “Get Key Settings”commands
    /// </summary>
    ONLYMASTERKEY = 0x00,

    /// <summary>
    /// “Get FID List”, “Get File Settings” and “Get Key Settings” commands succeed independentlyof a preceding application master key authentication (default)
    /// </summary>
    NOKEY = 0x02,
}

/// <summary>
/// Indicates use of 2 byte ISO/IEC 7816-4 File Identifies for files within the Application
/// </summary>
#[repr(u8)]
#[derive(Clone, Copy)]
pub enum FileIdentifiers {
    NOTUSED = 0x00,
    USED = 0x20,
}

#[repr(u8)]
#[derive(Clone, Copy)]
pub enum FileTypes {
    /// <summary>
    /// Standard Data File
    /// </summary>
    STANDARD = 0x00,

    /// <summary>
    /// Backup Data Files
    /// </summary>
    BACKUP = 0x01,

    /// <summary>
    /// Value Files with Backup
    /// </summary>
    VALUE = 0x02,

    /// <summary>
    /// Linear Record Files with Backup
    /// </summary>
    LINEARRECORD = 0x03,

    /// <summary>
    /// Cyclic Record Files with Backup
    /// </summary>
    CYCLICRECORD = 0x04,
}

mod apduinstructions;
mod apdustatuscodes;
pub mod desfire;
