use crate::Repo;
use std::io::Error;
use std::io::ErrorKind;
use std::process::Command;
use std::process::Output;

fn convert_result(status: Result<Output, Error>) -> Result<(), Error> {
    match status {
        Ok(output) => {
            if output.status.code() == Some(0) {
                Ok(())
            } else {
                Err(Error::new(
                    ErrorKind::Other,
                    std::str::from_utf8(&output.stderr).unwrap(),
                ))
            }
        }
        Err(e) => Err(e),
    }
}

/// Clone a repo to a directory and check out the branch.
/// The clone is shallow and only the needed branch is fetched.
pub fn clone(repo: &Repo) -> Result<(), Error> {
    println!("Cloning repository '{}'", repo.name);
    let status = Command::new("git")
        .arg("clone") // clone
        .arg(&repo.url) // this repo
        .arg(&repo.name) // to this dir
        .arg("--branch") // but only this branch
        .arg(&repo.branch)
        .arg("--depth") // and only to this depth
        .arg("1")
        .arg("--single-branch") // and no other branches
        .output();

    convert_result(status)
}

pub fn fetch(repo: &Repo) -> Result<(), Error> {
    println!("Fetching repository '{}'", repo.name);
    let status = Command::new("git")
        .arg("fetch") // checkout
        .current_dir(&repo.name)
        .output();

    convert_result(status)
}

pub fn pull(repo: &Repo) -> Result<(), Error> {
    println!("Pulling repository '{}'", repo.name);
    let status = Command::new("git")
        .arg("pull") // checkout
        .current_dir(&repo.name)
        .output();

    convert_result(status)
}

/// Checkout a branch and make sure it is up to date.
/// The checkout is shallow and only the needed branch is fetched.
pub fn checkout(repo: &Repo) -> Result<(), Error> {
    println!(
        "Checkout branch '{}' in repository '{}'",
        repo.branch, repo.name
    );
    let status = Command::new("git")
        .arg("checkout") // checkout
        .arg(&repo.branch) //this branch
        .current_dir(&repo.name)
        .output();

    convert_result(status)
}

pub fn apply(repo: &Repo) -> Result<(), Error> {
    let path = std::path::Path::new(repo.name.as_str());

    if !path.exists() {
        clone(repo)?;
    } else {
        // if exists
        fetch(repo)?;
        checkout(repo)?;
        pull(repo)?;
    }

    Ok(())
}

pub fn is_dirty(repo: &Repo) -> Result<bool, Error> {
    let status = Command::new("git")
        .arg("diff")
        .arg("--quiet")
        .current_dir(&repo.name)
        .output()?;
    Ok(status.status.success())
}

#[cfg(test)]
mod tests {

    #[test]
    fn test_clone_good_url() {
        let now = chrono::Utc::now().to_rfc3339();
        let dirname = now.as_str();

        let repo = crate::Repo {
            name: dirname.to_string(),
            url: "git@gitlab.motius.de:cluster-strategy/cluster-roadmap.git".to_string(),
            branch: "master".to_string(),
        };
        let r = crate::tool::git::clone(&repo);

        assert!(r.is_ok()); // this clone should work

        std::fs::remove_dir_all(dirname).expect("Failed to delete the test repo.");
    }

    #[test]
    fn test_clone_bad_url() {
        let now = chrono::Utc::now().to_rfc3339();
        let dirname = now.as_str();

        let repo = crate::Repo {
            name: dirname.to_string(),
            url: "git@gitlab.motius.de:cluster-strategy/nota_existo.git".to_string(),
            branch: "master".to_string(),
        };
        let r = crate::tool::git::clone(&repo);

        assert!(r.is_err()); // this clone should fail
    }

    #[test]
    fn test_apply_good_url() {
        let now = chrono::Utc::now().to_rfc3339();
        let dirname = now.as_str();

        let repo = crate::Repo {
            name: dirname.to_string(),
            url: "git@gitlab.motius.de:cluster-strategy/cluster-roadmap.git".to_string(),
            branch: "master".to_string(),
        };
        let r = crate::tool::git::apply(&repo);

        assert!(r.is_ok()); // this clone should work

        std::fs::remove_dir_all(dirname).expect("Failed to delete the test repo.");
    }

    #[test]
    fn test_apply_bad_url() {
        let now = chrono::Utc::now().to_rfc3339();
        let dirname = now.as_str();

        let repo = crate::Repo {
            name: dirname.to_string(),
            url: "git@gitlab.motius.de:cluster-strategy/nota_existo.git".to_string(),
            branch: "master".to_string(),
        };
        let r = crate::tool::git::apply(&repo);

        assert!(r.is_err()); // this clone should fail
    }
}
