use crate::{Config, CopyItem, DirType, Installah, PyProject, Repo, Service};
use eyre::{Context, ContextCompat, Result};
use rayon::prelude::*;
use std::io::{self, Write};
use std::path::Path;

use std::process::Command;

pub fn installah(base_path: &Path, force: bool) -> Result<()> {
    std::env::set_var("INSTALLAH_USER", whoami::username());
    std::env::set_var("INSTALLAH_UID", users::get_current_uid().to_string());

    let dirtype = if base_path.join("deployah.toml").exists() {
        DirType::Deployah
    } else if base_path.join("pyproject.toml").exists() {
        DirType::PyProject
    } else {
        DirType::None
    };

    match dirtype {
        DirType::Deployah => {
            let config: Config = toml::from_str(
                &std::fs::read_to_string(base_path.join("deployah.toml"))
                    .context("Reading deployah.toml")?,
            )
            .context("Deserializing deployah.toml")?;

            match &config.repos {
                None => println!(
                    "No repositories specified in '{}'",
                    base_path.to_str().unwrap_or_default()
                ),
                Some(repos) => {
                    handle_repos(repos, force)?;
                }
            }
        }
        DirType::PyProject => {
            let pypro: PyProject = toml::from_str(
                &std::fs::read_to_string(base_path.join("pyproject.toml"))
                    .context("Reading pyproject.toml")?,
            )
            .context("Deserializing pyproject.toml")?;

            match pypro.installah {
                None => println!(
                    "No suitable installah information in '{}'",
                    base_path.to_str().unwrap_or_default()
                ),
                Some(installah) => handle(installah, base_path, force)?,
            };
        }
        DirType::None => {
            println!(
                "Not a known directory type in '{}'",
                base_path.to_str().unwrap_or_default()
            )
        }
    }

    Ok(())
}

pub fn handle_repos(repos: &[Repo], force: bool) -> Result<()> {
    repos.par_iter().try_for_each(|repo_config| {
        let path = Path::new(repo_config.name.as_str());
        if path.exists() {
            installah(path, force)
        } else {
            println!("Repository does not '{}' exist", repo_config.name);
            Ok(())
        }
    })?;
    Ok(())
}

pub fn handle(installah: Installah, base_path: &Path, force: bool) -> Result<()> {
    // TODO this whole thing runs twice because of the privilege escalation

    println!("Install {:?}", &base_path);
    Command::new(format!(
        "/home/{}/.poetry/bin/poetry",
        std::env::var("INSTALLAH_USER").unwrap()
    ))
    .arg("install")
    .current_dir(&base_path)
    .output()
    .context("Failed to execute poetry")?;

    match installah.cp {
        None => (),
        Some(cp) => handle_cp(cp, base_path, force)?,
    }

    match installah.service {
        None => (),
        Some(service) => handle_service(service, base_path, force)?,
    }

    Ok(())
}

pub fn handle_service(service: Service, base_path: &Path, _force: bool) -> Result<()> {
    // TODO
    // resolve template vars
    // write all instances
    let template_content =
        std::fs::read_to_string(base_path.join(Path::new(&service.template))).unwrap_or_default();
    let service_content = resolve_specifiers(template_content, base_path);

    service.instance.par_iter().try_for_each(|instance| {
        println!("Set up service '{}'", instance);
        let location_str = format!(
            "/home/{}/.config/systemd/user.control/{}",
            std::env::var("INSTALLAH_USER").unwrap(),
            instance.as_str()
        );
        let location = Path::new(&location_str);

        std::fs::create_dir_all(
            Path::new(&location_str)
                .parent()
                .with_context(|| format!("Could not get path '{}'", &location_str))?,
        )
        .with_context(|| format!("Could not create path '{}'", &location_str))?;

        // TODO: check if already exists
        // stop, write, daemon-reload and restart if exists
        // if not exists, write and enable
        // also, 0 style-points for calling shell commands
        std::fs::write(location, &service_content).with_context(|| {
            format!(
                "Could not write to file '{}'",
                location.to_str().unwrap_or_default()
            )
        })?;

        Command::new("systemctl")
            .arg("--user")
            .arg("enable")
            .arg(instance)
            .arg("--now")
            .output()
            .context("Failed to execute systemctl")?;

        Command::new("systemctl")
            .arg("--user")
            .arg("daemon-reload")
            .output()
            .context("Failed to execute systemctl")?;

        Ok::<_, eyre::Error>(())
    })?;

    Ok(())
}

pub fn resolve_specifiers(template_content: String, base_path: &Path) -> String {
    // TODO: make this less dependent on where the tool is run from
    // TODO: make it easier to add more specifiers
    // TODO: this is all wrong
    let mut curr_dir = std::env::current_dir().unwrap_or_default();
    let mut _resolved_content = String::new();
    curr_dir.push(base_path);
    _resolved_content =
        template_content.replace("%repo_root", curr_dir.to_str().unwrap_or_default());
    curr_dir.pop(); // this modifies curr_dir, I'd rather create a new Path instead
    curr_dir.pop(); // this modifies curr_dir, I'd rather create a new Path instead
    _resolved_content =
        _resolved_content.replace("%deployah_root", curr_dir.to_str().unwrap_or_default());
    _resolved_content = _resolved_content.replace(
        "%installah_user",
        &std::env::var("INSTALLAH_USER").unwrap_or_default(),
    );
    _resolved_content =
        _resolved_content.replace("%uid", &std::env::var("INSTALLAH_UID").unwrap_or_default());
    _resolved_content
}

pub fn handle_cp(cp: Vec<CopyItem>, base_path: &Path, force: bool) -> Result<()> {
    cp.par_iter().try_for_each(|cpi| {
        let destination = resolve_specifiers(cpi.dest.to_owned(), base_path);
        println!("Copy '{}' to '{}'", &cpi.src, &destination);

        std::fs::create_dir_all(
            Path::new(&destination)
                .parent()
                .with_context(|| format!("Could not get path '{}'", &destination))?,
        )
        .with_context(|| format!("Could not create path '{}'", &destination))?;

        if !Path::new(&destination).exists() || force {
            std::fs::copy(base_path.join(Path::new(&cpi.src)), &destination).with_context(
                || format!("Could not copy from '{}' to '{}'", &cpi.src, &destination),
            )?;
        } else {
            print!("Config file already exists, do you want to overwrite? [yes/no] ");
            io::stdout().flush()?;
            let answer: String = text_io::read!();
            if answer == "yes" {
                std::fs::copy(base_path.join(Path::new(&cpi.src)), &destination).with_context(
                    || format!("Could not copy from '{}' to '{}'", &cpi.src, &destination),
                )?;
            }
        }

        Ok::<_, eyre::Error>(())
    })?;
    Ok(())
}
