use clap::{
    crate_authors, crate_description, crate_name, crate_version, App, AppSettings, SubCommand,
};
use deployah::subcommand;
use eyre::Result;

fn main() -> Result<()> {
    let matches = App::new(crate_name!())
        .setting(AppSettings::ArgRequiredElseHelp)
        .version(crate_version!())
        .author(crate_authors!())
        .about(crate_description!())
        .args_from_usage(
            "--config=[FILE] 'Set a custom config file'
             -v...           'Set the level of verbosity'",
        )
        .subcommand(
            SubCommand::with_name("init")
                .about("Initialize a deployah directory")
                .args_from_usage(
                    "--title=[TITLE] 'Set a title for the config file'
                     --name=[NAME]   'Set a project name'",
                ),
        )
        .subcommand(
            SubCommand::with_name("add")
                .about("Add a repository to the config")
                .args_from_usage(
                    "--name=[NAME]     'Set the name of the repository'
                     --url=[REPOURL]   'Set the URL of the repository'
                     --branch=[BRANCH] 'Set the branch of the repository'",
                ),
        )
        .subcommand(
            SubCommand::with_name("update")
                .about("Update repositories as specified in deployah.toml"),
        )
        .subcommand(
            SubCommand::with_name("purge")
                .about("Purges all repositories specified in deployah.toml"),
        )
        .subcommand(
            SubCommand::with_name("installah")
                .about("Install a repository based on installah data in pyproject.toml")
                .args_from_usage(
                    "--force 'Force overwrite of existing files'
                     [PATH]  'Set the path of the repository which should be installed'",
                ),
        )
        .get_matches();

    if let Some(matches) = matches.subcommand_matches("init") {
        subcommand::init::init(
            matches.value_of("title").unwrap_or_default(),
            matches.value_of("name").unwrap_or_default(),
        )?;
    }

    if let Some(matches) = matches.subcommand_matches("add") {
        subcommand::add::add(
            matches.value_of("name").unwrap_or_default(),
            matches.value_of("url").unwrap_or_default(),
            matches.value_of("branch").unwrap_or_default(),
        )?
    }

    if let Some(_matches) = matches.subcommand_matches("update") {
        subcommand::update::update()?;
    }

    if let Some(_matches) = matches.subcommand_matches("purge") {
        subcommand::purge::purge()?;
    }

    if let Some(matches) = matches.subcommand_matches("installah") {
        subcommand::installah::installah(
            std::path::Path::new(matches.value_of("PATH").unwrap_or(".")),
            matches.is_present("force"),
        )?;
    }
    Ok(())
}
