# dependency-graph
Dependency-graph is a minimal library, exposing `DependencyGraph` structure and a single `Node` trait.

To use the library, simply implement the `Node` trait's two functions for the object you wish to.

# Example
In this example, we'll be using `dependency-graph` to resolve dependencies between our `Package` structs. We'll be using a custom `Dependency` type, because we want to include Semantic Versioning constraints in our dependencies. That way we'll be able to say that our package `mypackage` depends on version 2.0 of some package `some-library` for instance.

Our `Package` and `Dependency` structs use the `semver::Version` and `semver::VersionReq` types to define the versions of the `Packages` (such as 1.2.3-beta.4) and the dependency requirements such as `>=2.0`. See the [semver](https://crates.io/crates/semver) crate for more information.

First we define the `Package` struct:
```rust
struct Package {
    name: &'static str,
    version: Version,
    dependencies: Vec<Dependency>,
}
```
Where `Dependency` is:
```rust
struct Dependency {
    name: &'static str,
    version: VersionReq,
}

```

Implementing the `Node` trait for our `Package` is pretty simple:
```rust
impl Node for Package {
    type DependencyType = Dependency;

    fn dependencies(&self) -> &[Self::DependencyType] {
        &self.dependencies[..]
    }

    fn matches(&self, dependency: &Self::DependencyType) -> bool {
        // Check that name is an exact match, and that the dependency
        // requirements are fulfilled by our own version
        self.name == dependency.name && dependency.version.matches(&self.version)
    }
}
```

