// Copyright 2020-2021 the Deno authors. All rights reserved. MIT license.
use super::{Context, LintRule, DUMMY_NODE};
use crate::swc_util::StringRepr;
use crate::ProgramRef;
use swc_common::Spanned;
use swc_ecmascript::ast::BinExpr;
use swc_ecmascript::ast::BinaryOp::{EqEq, EqEqEq, NotEq, NotEqEq};
use swc_ecmascript::ast::Expr::{Lit, Tpl, Unary};
use swc_ecmascript::ast::Lit::Str;
use swc_ecmascript::ast::UnaryOp::TypeOf;
use swc_ecmascript::visit::{noop_visit_type, Node, Visit};

pub struct ValidTypeof;

const CODE: &str = "valid-typeof";
const MESSAGE: &str = "Invalid typeof comparison value";

impl LintRule for ValidTypeof {
  fn new() -> Box<Self> {
    Box::new(ValidTypeof)
  }

  fn tags(&self) -> &'static [&'static str] {
    &["recommended"]
  }

  fn code(&self) -> &'static str {
    CODE
  }

  fn lint_program(&self, context: &mut Context, program: ProgramRef) {
    let mut visitor = ValidTypeofVisitor::new(context);
    match program {
      ProgramRef::Module(m) => visitor.visit_module(m, &DUMMY_NODE),
      ProgramRef::Script(s) => visitor.visit_script(s, &DUMMY_NODE),
    }
  }

  #[cfg(feature = "docs")]
  fn docs(&self) -> &'static str {
    include_str!("../../docs/rules/valid_typeof.md")
  }
}

struct ValidTypeofVisitor<'c, 'view> {
  context: &'c mut Context<'view>,
}

impl<'c, 'view> ValidTypeofVisitor<'c, 'view> {
  fn new(context: &'c mut Context<'view>) -> Self {
    Self { context }
  }
}

impl<'c, 'view> Visit for ValidTypeofVisitor<'c, 'view> {
  noop_visit_type!();

  fn visit_bin_expr(&mut self, bin_expr: &BinExpr, _parent: &dyn Node) {
    if !bin_expr.is_eq_expr() {
      return;
    }

    match (&*bin_expr.left, &*bin_expr.right) {
      (Unary(unary), operand) | (operand, Unary(unary))
        if unary.op == TypeOf =>
      {
        match operand {
          Unary(unary) if unary.op == TypeOf => {}
          Lit(Str(str)) => {
            if !is_valid_typeof_string(&str.value) {
              self.context.add_diagnostic(str.span, CODE, MESSAGE);
            }
          }
          Tpl(tpl) => {
            if tpl
              .string_repr()
              .map_or(false, |s| !is_valid_typeof_string(&s))
            {
              self.context.add_diagnostic(tpl.span, CODE, MESSAGE);
            }
          }
          _ => {
            self.context.add_diagnostic(operand.span(), CODE, MESSAGE);
          }
        }
      }
      _ => {}
    }
  }
}

fn is_valid_typeof_string(str: &str) -> bool {
  matches!(
    str,
    "undefined"
      | "object"
      | "boolean"
      | "number"
      | "string"
      | "function"
      | "symbol"
      | "bigint"
  )
}

trait EqExpr {
  fn is_eq_expr(&self) -> bool;
}

impl EqExpr for BinExpr {
  fn is_eq_expr(&self) -> bool {
    matches!(self.op, EqEq | NotEq | EqEqEq | NotEqEq)
  }
}

#[cfg(test)]
mod tests {
  use super::*;

  #[test]
  fn valid_typeof_valid() {
    assert_lint_ok! {
      ValidTypeof,
      r#"typeof foo === "undefined""#,
      r#"typeof foo === "object""#,
      r#"typeof foo === "boolean""#,
      r#"typeof foo === "number""#,
      r#"typeof foo === "string""#,
      r#"typeof foo === "function""#,
      r#"typeof foo === "symbol""#,
      r#"typeof foo === "bigint""#,

      r#"typeof foo == 'undefined'"#,
      r#"typeof foo == 'object'"#,
      r#"typeof foo == 'boolean'"#,
      r#"typeof foo == 'number'"#,
      r#"typeof foo == 'string'"#,
      r#"typeof foo == 'function'"#,
      r#"typeof foo == 'symbol'"#,
      r#"typeof foo == 'bigint'"#,

      // https://github.com/denoland/deno_lint/issues/741
      r#"typeof foo !== `undefined`"#,
      r#"typeof foo !== `object`"#,
      r#"typeof foo !== `boolean`"#,
      r#"typeof foo !== `number`"#,
      r#"typeof foo !== `string`"#,
      r#"typeof foo !== `function`"#,
      r#"typeof foo !== `symbol`"#,
      r#"typeof foo !== `bigint`"#,

      r#"typeof bar != typeof qux"#,
    };
  }

  #[test]
  fn valid_typeof_invalid() {
    assert_lint_err! {
      ValidTypeof,
      r#"typeof foo === "strnig""#: [{
        col: 15,
        message: MESSAGE
      }],
      r#"typeof foo == "undefimed""#: [{
        col: 14,
        message: MESSAGE
      }],
      r#"typeof bar != "nunber""#: [{
        col: 14,
        message: MESSAGE
      }],
      r#"typeof bar !== "fucntion""#: [{
        col: 15,
        message: MESSAGE
      }],
      r#"typeof foo === undefined"#: [{
        col: 15,
        message: MESSAGE
      }],
      r#"typeof bar == Object"#: [{
        col: 14,
        message: MESSAGE
      }],
      r#"typeof baz === anotherVariable"#: [{
        col: 15,
        message: MESSAGE
      }],
    }
  }
}
