//! Error types

use abscissa_core::error::{BoxError, Context};
use std::{
    fmt::{self, Display},
    io,
    ops::Deref,
};
use thiserror::Error;

/// Kinds of errors
#[derive(Copy, Clone, Debug, Eq, Error, PartialEq)]
pub enum ErrorKind {
    /// Error in configuration file
    #[error("config error")]
    Config,

    /// Currency-related error
    #[error("currency error")]
    Currency,

    /// HTTP errors
    #[error("HTTP error")]
    Http,

    /// Input/output error
    #[error("I/O error")]
    Io,

    /// Parse errors
    #[error("parse error")]
    Parse,

    /// Source errors
    #[error("source")]
    Source,
}

impl ErrorKind {
    /// Create an error context from this error
    pub fn context(self, source: impl Into<BoxError>) -> Context<ErrorKind> {
        Context::new(self, Some(source.into()))
    }
}

/// Error type
#[derive(Debug)]
pub struct Error(Box<Context<ErrorKind>>);

impl Deref for Error {
    type Target = Context<ErrorKind>;

    fn deref(&self) -> &Context<ErrorKind> {
        &self.0
    }
}

impl Display for Error {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        self.0.fmt(f)
    }
}

impl std::error::Error for Error {
    fn source(&self) -> Option<&(dyn std::error::Error + 'static)> {
        self.0.source()
    }
}

impl From<ErrorKind> for Error {
    fn from(kind: ErrorKind) -> Self {
        Context::new(kind, None).into()
    }
}

impl From<Context<ErrorKind>> for Error {
    fn from(context: Context<ErrorKind>) -> Self {
        Error(Box::new(context))
    }
}

impl From<http::Error> for Error {
    fn from(err: http::Error) -> Self {
        ErrorKind::Http.context(err).into()
    }
}

impl From<hyper::Error> for Error {
    fn from(err: hyper::Error) -> Self {
        ErrorKind::Http.context(err).into()
    }
}

impl From<io::Error> for Error {
    fn from(err: io::Error) -> Self {
        ErrorKind::Io.context(err).into()
    }
}

impl From<serde_json::Error> for Error {
    fn from(err: serde_json::Error) -> Self {
        ErrorKind::Parse.context(err).into()
    }
}

impl From<rust_decimal::Error> for Error {
    fn from(err: rust_decimal::Error) -> Self {
        ErrorKind::Parse.context(err).into()
    }
}

impl From<stdtx::Error> for Error {
    fn from(err: stdtx::Error) -> Error {
        Context::new(ErrorKind::Parse, Some(err.into())).into()
    }
}
