/*
 * database.rs
 *
 * DEEPWELL - Wikijump API provider and database manager
 * Copyright (C) 2019-2022 Wikijump Team
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

use anyhow::Result;
use sea_orm::{ConnectOptions, Database, DatabaseConnection};
use sqlx::{Pool, Postgres};
use std::time::Duration;

pub async fn connect<S: Into<String>>(database_uri: S) -> Result<DatabaseConnection> {
    let mut options = ConnectOptions::new(database_uri.into());
    options
        .min_connections(4)
        .max_connections(100)
        .connect_timeout(Duration::from_secs(5))
        .idle_timeout(Duration::from_secs(10))
        .sqlx_logging(true);

    let db = Database::connect(options).await?;
    Ok(db)
}

pub async fn migrate(database_uri: &str) -> Result<()> {
    let pool = Pool::<Postgres>::connect(database_uri).await?;

    sqlx::migrate!("./migrations").run(&pool).await?;
    Ok(())
}
