/*!
# A Procedural Macro for Decorators
The Decorator macro generates a decorator method for each marked field. A field can be marked with the `#[dec]` field in front.
## Example
```
#[derive(Decorator)]
struct Widget {
    #[dec]
    width: u32,
    #[dec]
    height: u32,
}
```
Generates into:
```
struct Widget {
    width: u32,
    height: u32,
}
impl Widget {
    pub fn width(self, width: u32) -> Self {
        Self {
            width,
            ..self
        }
    }
    pub fn height(self, height: u32) -> Self {
        Self {
            height,
            ..self
        }
    }
}
```
Which can be used like:
```
let w = some_widget.width(10).height(20);
assert_eq!(w, Widget {width: 10, height: 20});
```
*/

use proc_macro::TokenStream;
use quote::quote;
use syn::{parse_macro_input, ItemStruct};

/// A macro to generate a decorator method for each marked field.
/// A field can be marked with the `#[dec]` field in front.
#[proc_macro_derive(Decorator, attributes(dec))]
pub fn derive_decorator(input: TokenStream) -> TokenStream {
    let item_struct = parse_macro_input!(input as ItemStruct);

    let mut decorators: Vec<(proc_macro2::Ident, syn::Type)> = vec![];
    if let syn::Fields::Named(ref fields_named) = item_struct.fields {
        for field in fields_named.named.iter() {
            for attr in field.attrs.iter() {
                match attr.parse_meta().unwrap() {
                    syn::Meta::Path(ref path) if *path.get_ident().unwrap() == "dec" => {
                        let item = field.clone();
                        decorators.push((item.ident.unwrap(), item.ty))
                    }
                    _ => (),
                }
            }
        }
    }

    let decorators = decorators.iter().fold(quote!(), |accumulator, (name, ty)| {
        quote! {
            #accumulator
            pub fn #name(self, #name:#ty ) -> Self {
                Self {
                    #name,
                    ..self
                }
            }
        }
    });

    let name = item_struct.ident;
    let (impl_generics, ty_generics, where_clause) = item_struct.generics.split_for_impl();
    TokenStream::from(quote! {
            impl #impl_generics #name #ty_generics #where_clause  {
                #decorators
            }
    })
}
