extern crate crypto;
use self::crypto::digest::Digest;
use self::crypto::sha1::Sha1;
use num_bigint::BigUint;
use serde::{Deserialize, Serialize};
use serde_json::{Error, Result as JsonResult, Value};
use std::collections::HashMap;
use std::fmt;
use std::fs::File;
use std::io::{prelude::*, BufReader};

// Decider is a library for determining whether a given feature should be available
// in a given context.  It supports a very expressive set of operations for determining
// which features should be available (and in what variant, if more than one is desired)
// in a context.

// A context captures the relevant state in which we want to find out whether a feature
// should be available.
#[derive(Serialize, Deserialize, Debug)]
pub struct Context {
    pub user_id: i64, // FIXME: make this u64 after figuring out how to make range literals unsigned.
    /* IETF language tag representing the preferred locale for
     * the client, used for providing localized content. Consists of
     * an ISO 639-1 primary language subtag and an optional
     * ISO 3166-1 alpha-2 region subtag separated by an underscore.
     * e.g. en, en_US
     */
    pub locale: Option<String>,
    pub geolocation: Option<String>, // A two-character ISO 3166-1 country code
    pub device_id: Option<String>,
    pub origin_service: Option<String>,
    pub other_data: Option<HashMap<String, String>>,
}

impl Context {
    fn get_field(&self, field: &String) -> Option<String> {
        // FIXME: deal with access for other fields
        if field == "user_id" {
            Some(self.user_id.to_string())
        } else {
            return None;
        }
    }

    fn cmp(&self, field: &String, value: &String) -> bool {
        // FIXME: update the type sig of this to signal missing fields
        let fo = self.get_field(field);
        match fo {
            None => false,
            Some(s) => &s == value,
        }
    }

    fn cmp_op(&self, comp: Comp, field: &String, value: &String) -> bool {
        let fo = self.get_field(field);
        match fo {
            None => false,
            Some(s) => {
                // FIXME: get rid of the unwrap!
                let int_val: i64 = s.parse().unwrap();
                let value_val: i64 = value.parse().unwrap();
                match comp {
                    Comp::GT => int_val > value_val,
                    Comp::LT => int_val < value_val,
                    Comp::GE => int_val >= value_val,
                    Comp::LE => int_val <= value_val,
                }
            }
        }
    }
}

// Features represent a unit of controllable code, that we might want to turn off, make
// available only to some users, etc..  This is what FeatureFlags/AB tests control.
#[derive(Serialize, Deserialize, Debug, Clone)]
pub struct Feature {
    id: u32,
    name: String,
    enabled: bool,
    start_ts: u32, // TODO: consider whether this should just be created_at
    stop_ts: u32,  // TODO: should we get rid of a version by creating a new one?
    version: u32,
    shuffle_version: u32,
    variants: Vec<Variant>,
    platform_bitmask: u128, //TODO: should this be made a vec for unbounded size?
    targeting: Option<TargetingTree>,
    overrides: Option<HashMap<String, TargetingTree>>,
}

// Variants are primarily used for fractional availability, where, eg., 10% of users
// are exposed to a given version of a feature.
#[derive(Serialize, Deserialize, Debug, Clone)]
pub struct Variant {
    name: String,
    #[serde(rename = "range_start")] // TODO change to alias and remove
    lo: f32, // consider changing away from floats.
    #[serde(rename = "range_end")]
    hi: f32,
}

#[derive(Serialize, Deserialize, Debug, Clone)]
pub enum EqSwitch {
    EqOne { field: String, value: String },
    EqMany { field: String, values: Vec<String> },
}

#[derive(Serialize, Deserialize, Debug, Clone)]
pub struct EqStruct {
    field: String,
    value: Option<String>,
    values: Option<Vec<String>>,
}

impl EqStruct {
    fn eval(&self, ctx: &Context) -> bool {
        // FIXME: this makes values trump value, which is gross.
        println!("ctx: {:#?}, tt: {:#?}", ctx, self);
        return match &self.values {
            Some(vl) => vl.iter().any(|x| ctx.cmp(&self.field, &x)),
            None => match &self.value {
                None => true,
                Some(v) => ctx.cmp(&self.field, &v),
            },
        };
    }
}

// The TargetingTree allows for arbitrary targeting operations, which can be used to
// limit an experiment to employees, or users from new zealand, only to iOS users, or
// to users in New Zealand, etc..
#[derive(Serialize, Deserialize, Debug, Clone)]
pub enum TargetingTree {
    ALL(Vec<TargetingTree>),
    ANY(Vec<TargetingTree>),
    NOT(Box<TargetingTree>),
    //#[serde(flatten)]
    EQ(EqStruct),
    GT { field: String, value: String },
    LT { field: String, value: String },
    GE { field: String, value: String },
    LE { field: String, value: String },
    NE { field: String, value: String },
}

enum Comp {
    GT,
    LT,
    GE,
    LE,
}

impl TargetingTree {
    fn eval(&self, ctx: &Context) -> bool {
        match self {
            TargetingTree::ALL(xs) => xs.iter().all(|x| x.eval(&ctx)),
            TargetingTree::ANY(xs) => xs.iter().any(|x| x.eval(&ctx)),
            TargetingTree::NOT(x) => !x.eval(&ctx),
            TargetingTree::EQ(es) => es.eval(&ctx),
            TargetingTree::GT { field, value } => ctx.cmp_op(Comp::GT, field, value),
            TargetingTree::LT { field, value } => ctx.cmp_op(Comp::LT, field, value),
            TargetingTree::GE { field, value } => ctx.cmp_op(Comp::GE, field, value),
            TargetingTree::LE { field, value } => ctx.cmp_op(Comp::LE, field, value),
            TargetingTree::NE { field, value } => !ctx.cmp(&field, value),
        }
    }
}
// Decider determines feature availability by chaining together simple
// decisionmaker functions to enable complicated logic to be expressed as a
// composition of simple, testable functions. Those compositions are tested
// against features, an abstsraction for bits of code that enabling feature
// flagging, AB testing, etc..

pub struct Decider {
    features: Vec<Feature>,
    decisionmakers:
        Vec<fn(f: &Feature, ctx: &Context) -> Result<DecisionmakerResult, DeciderFailType>>,
}

impl fmt::Debug for Decider {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, "Decider: {:#?}", self.features)
    }
}

impl Decider {
    pub fn choose(
        &self,
        feature_name: String,
        ctx: &Context,
    ) -> Result<Option<Decision>, DeciderFailType> {
        let f = self.feature_by_name(feature_name)?;
        return self.decide(&f, ctx);
    }

    fn feature_by_name(&self, feature_name: String) -> Result<Feature, DeciderFailType> {
        let fo = self.features.iter().find(|f| f.name == feature_name);
        return match fo {
            None => Err(DeciderFailType::FeatureNotFound),
            Some(feature) => Ok(feature.clone()),
        };
    }

    fn decide(&self, f: &Feature, ctx: &Context) -> Result<Option<Decision>, DeciderFailType> {
        // TODO: decide whether decide should return a Result<Option<Decision>>, to
        // account for ill-defined Features or missing data on the Context. Probably
        // yes, but currently Decision also has an option on name/bucketing.  Maybe
        // decide should return a Result<Decision>?
        println!(
            "calling decide on feature name: {:#?} with context:{:#?}",
            f.name, ctx
        );

        for fun in &self.decisionmakers {
            let res = fun(&f, &ctx)?; // Should I let errors bubble up?  probably not.
            match res {
                DecisionmakerResult::None => return Ok(None),
                DecisionmakerResult::Pass(s) => {
                    println!("decisionmaker passed: {}", s);
                    ()
                }
                DecisionmakerResult::Decided(d) => {
                    println!("got a decision:{:#?}", d);
                    return Ok(Some(d));
                }
            }
        }
        return Ok(None); // default open (just return nothing)
    }
}

impl Feature {
    fn bucketing_string(&self, ctx: &Context) -> String {
        return format!("{}.{}.{}", self.name, self.shuffle_version, ctx.user_id);
    }

    fn decision_at(&self, i: i32) -> Option<Decision> {
        let f = (i as f32) / 1000.0;
        let v = self.variants.iter().find(|x| x.lo <= f && x.hi > f);
        return match v {
            None => None,
            Some(variant) => Some(Decision {
                name: variant.name.clone(),
                bucket: Some(i),
                emit_event: true,
            }),
        };
    }
}

pub fn darkmode(f: &Feature, _ctx: &Context) -> Result<DecisionmakerResult, DeciderFailType> {
    if f.enabled {
        return Ok(DecisionmakerResult::Pass("darkmode:enabled".to_string()));
    } else {
        return Ok(DecisionmakerResult::None);
    }
}

pub fn locale(f: &Feature, _ctx: &Context) -> Result<DecisionmakerResult, DeciderFailType> {
    if !f.enabled {
        // TODO: add locales to Feature, compare ctx.locale to that
        return Ok(DecisionmakerResult::Pass("locale:fixme".to_string()));
    } else {
        return Ok(DecisionmakerResult::Pass("locale:fixme".to_string()));
    }
}

pub fn fractional_availability(
    f: &Feature,
    ctx: &Context,
) -> Result<DecisionmakerResult, DeciderFailType> {
    let b_str = f.bucketing_string(ctx);
    let i = bucket(b_str);
    let flt = (i as f32) / 1000.0;
    let v = f.variants.iter().find(|x| x.lo <= flt && x.hi > flt);
    return match v {
        None => Ok(DecisionmakerResult::Pass(
            "frac_avail:not in variant".to_string(),
        )),
        Some(variant) => Ok(DecisionmakerResult::Decided(Decision {
            name: variant.name.clone(),
            bucket: Some(i),
            emit_event: false,
        })),
    };
}

#[derive(Serialize, Deserialize, Debug)]
pub enum DecisionmakerResult {
    // TODO: find a better nome
    Pass(String),      // I didn't make a decision because...
    None,              // response is nothing
    Decided(Decision), // actual decision.
}

#[derive(Debug)]
pub enum DeciderFailType {
    FeatureNotFound,
    InvalidFeature,
    InvalidDecisionMaker(String),
    IoError(std::io::Error),
    SerdeError(serde_json::Error),
}

impl From<std::io::Error> for DeciderFailType {
    fn from(e: std::io::Error) -> DeciderFailType {
        DeciderFailType::IoError(e)
    }
}

impl From<serde_json::Error> for DeciderFailType {
    fn from(e: serde_json::Error) -> DeciderFailType {
        DeciderFailType::SerdeError(e)
    }
}

#[derive(PartialEq, Eq, Serialize, Deserialize, Debug)]
pub struct Decision {
    name: String,
    emit_event: bool,
    bucket: Option<i32>,
}

fn bucket(bucketing_str: String) -> i32 {
    // FIXME: take in number of buckets as a param.
    let mut hasher = Sha1::new();
    hasher.input_str(&bucketing_str);
    let bigint = BigUint::parse_bytes(hasher.result_str().as_bytes(), 16);
    let res = match bigint {
        Some(v) => v % 1000u32,
        None => BigUint::from(9999u32), // FIXME: don'T use sentinel out-of-range values to indicate error.
    };
    let n_as_i32: i32 = res.try_into().unwrap(); // FIXME: get rid of the unwrap
    return n_as_i32;
}

type Decisionmaker = fn(f: &Feature, ctx: &Context) -> Result<DecisionmakerResult, DeciderFailType>;

fn name_to_decisionmaker(name: &str) -> Result<Decisionmaker, String> {
    return match name {
        "darkmode" => Ok(darkmode),
        "locale" => Ok(locale),
        "fractional_availability" => Ok(fractional_availability),
        _ => Err(format!("Invalid decisionmaker name: {:?}", name)),
    };
}

fn string_to_decisionmakers(cfg: String) -> Result<Vec<Decisionmaker>, String> {
    let dml = cfg
        .split_whitespace()
        .flat_map(|s| name_to_decisionmaker(s))
        .take(5)
        .collect();
    return Ok(dml);
}

pub fn init(cfg: String, features: Vec<Feature>) -> Result<Decider, DeciderFailType> {
    let dml = string_to_decisionmakers(cfg);
    return match dml {
        Ok(decisionmakers) => Ok(Decider {
            features,
            decisionmakers,
        }),
        Err(s) => Err(DeciderFailType::InvalidDecisionMaker(s)),
    };
}

// This section is legacy code to deal with reddit's existing experiment format.
pub fn init_decider(cfg: String, filepath: String) -> Result<Decider, DeciderFailType> {
    let file = File::open(filepath)?;
    let reader = BufReader::new(file);
    let ec: ExperimentConfig = serde_json::from_reader(reader)?;
    let fl: Vec<Feature> = ec
        .into_values()
        .map(|exp| experiment_to_feature(&exp))
        .collect();

    init(cfg, fl)
}

type ExperimentConfig = HashMap<String, Experiment>;

#[derive(Serialize, Deserialize, Debug)]
struct Experiment {
    id: u32,
    name: String,
    enabled: bool,
    version: String,
    r#type: ExperimentType,
    start_ts: u32,
    stop_ts: u32,
    experiment: InnerExperiment,
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all(deserialize = "snake_case"))]
enum ExperimentType {
    RangeVariant,
    FeatureRollout, // FIXME: get rid of this after the great RangeVariant takeover.
}

#[derive(Serialize, Deserialize, Debug)]
struct InnerExperiment {
    // FIXME: better name, plzkkthxbai?
    variants: Vec<Variant>, // TODO: figure out how to make a variable-length array in a struct, maybe?
    experiment_version: u32,
    shuffle_version: u32,
    bucket_val: String,
    overrides: Option<HashMap<String, TargetingTree>>,
    targeting: Option<TargetingTree>,
}

fn experiment_to_feature(exp: &Experiment) -> Feature {
    return Feature {
        // FIXME: surely there must be a better way
        id: exp.id,
        name: exp.name.clone(),
        enabled: exp.enabled,
        start_ts: exp.start_ts,
        stop_ts: exp.stop_ts,
        version: exp.experiment.experiment_version,
        shuffle_version: exp.experiment.shuffle_version,
        variants: exp.experiment.variants.clone(),
        targeting: exp.experiment.targeting.clone(),
        overrides: exp.experiment.overrides.clone(),
        platform_bitmask: 0,
    };
}

// here for integration testing of the workspace, remove before first commit.
pub fn add_one(x: i32) -> i32 {
    x + 1
}

#[cfg(test)]
mod tests {
    use super::*;
    fn make_feature() -> Feature {
        let v = Variant {
            name: "enabled".to_string(),
            lo: 0.0,
            hi: 0.1,
        };
        Feature {
            id: 1,
            name: "first_feature".to_string(),
            enabled: true,
            start_ts: 0,
            stop_ts: 2147483648, // 2 ** 31 - far future.
            version: 1,
            shuffle_version: 0,
            variants: vec![v],
            platform_bitmask: 0,
            targeting: None,
            overrides: None,
        }
    }

    fn make_ctx(json: Option<String>) -> Result<Context, Error> {
        let c = Context {
            user_id: 795244,
            locale: Some("US".to_string()),
            device_id: None,
            geolocation: None,
            origin_service: None,
            other_data: None,
        };
        return match json {
            None => Ok(c),
            Some(s) => {
                let cr = serde_json::from_str(&s);
                cr
            }
        };
    }

    fn make_tt(json: String) -> TargetingTree {
        let res: TargetingTree = serde_json::from_str(&json).unwrap();
        res
    }

    #[test]
    fn parse_targeting_tree() {
        let data2 = r#"
{"NE": {"field": "user_id", "value": "795244"}}
"#;
        let data3 = r#"
{"NE": {"field": "user_id", "value": "7"}}
"#;

        let tt1 = make_tt(r#"{"EQ": {"field": "user_id", "values": ["795244"]}}"#.to_string());
        let tt2 = make_tt(r#"{"EQ": {"field": "user_id", "value": "795244"}}"#.to_string());
        let tt3 = make_tt(r#"{"NE": {"field": "user_id", "value": "795244"}}"#.to_string());
        let tt4 = make_tt(r#"{"GT": {"field": "user_id", "value": "7"}}"#.to_string());
        let tt5 = make_tt(r#"{"LT": {"field": "user_id", "value": "8"}}"#.to_string());
        let tt6 = make_tt(r#"{"GE": {"field": "user_id", "value": "8"}}"#.to_string());
        let tt7 = make_tt(r#"{"LE": {"field": "user_id", "value": "8"}}"#.to_string());

        let ctx1 = make_ctx(None).unwrap();
        let ctx2: Context = make_ctx(Some(r#"{"user_id": 7}"#.to_string())).unwrap();

        assert!(tt1.eval(&ctx1)); // ctx1 has user_id 795244, so EQ passes
        assert!(!tt1.eval(&ctx2)); // ctx2 has user_id 7, so EQ fails
        assert!(tt2.eval(&ctx1)); // tt1 and tt2 are the same, just different representations
        assert!(!tt2.eval(&ctx2)); // so should have same results

        assert!(!tt3.eval(&ctx1)); // ctx has user_id 795244, so NE against 795244 fails
        assert!(tt3.eval(&ctx2)); // ctx2 has user_id 7, so NE against 7 passes

        assert!(tt4.eval(&ctx1)); // ctx has user_id 795244, so GT against 7 passes
        assert!(!tt4.eval(&ctx2)); // ctx2 has user_id 7, so GT against 7 fails

        assert!(!tt5.eval(&ctx1)); // ctx has user_id 795244, so LT against 8 passes
        assert!(tt5.eval(&ctx2)); // ctx2 has user_id 7, so LT against 8 fails

        assert!(tt6.eval(&ctx1)); // ctx has user_id 795244, so GE against 8 passes
        assert!(!tt6.eval(&ctx2)); // ctx2 has user_id 7, so GE against 8 fails

        assert!(!tt7.eval(&ctx1)); // ctx has user_id 795244, so LE against 8 fails
        assert!(tt7.eval(&ctx2)); // ctx2 has user_id 7, so LE against 8 passes
    }

    fn build_decider(
        cfgo: Option<String>,
        fvo: Option<Vec<Feature>>,
    ) -> Result<Decider, DeciderFailType> {
        let fv = match fvo {
            None => vec![make_feature()],
            Some(fv) => fv,
        };
        let cfg = match cfgo {
            Some(s) => s,
            None => "darkmode".to_string(), // set a reasonable default here?
        };
        return init(cfg, fv);
    }

    #[test]
    fn decider_initialization_works() -> Result<(), DeciderFailType> {
        let ctx = make_ctx(None)?;
        let d = build_decider(None, None)?;
        d.choose("first_feature".to_string(), &ctx)?;
        Ok(())
    }

    #[test]
    fn decider_errors_on_missing_name() -> Result<(), DeciderFailType> {
        let ctx = make_ctx(None)?;
        let d = build_decider(None, None)?;
        let r = d.choose("missing".to_string(), &ctx);
        match r {
            Ok(_) => panic!(),
            Err(e) => println!("got expected error: {:#?}", e),
        }
        Ok(())
    }

    #[test]
    fn another_test() {}
}
