#![feature(binary_heap_retain)]

//! Data structures and helpers for *debouncing* a stream of events: removing
//! duplicate events occurring closely in time.
//!
//! Examples of such events:
//!
//! - File write operations coming from a file system monitor
//! - Similar log strings from a `tail -f`
//! - Hardware signals from a sensor
//!
//! Debouncers accept a raw stream of events and add a grace period during which
//! similar events are considered duplicates. The event is delivered after the
//! timeout expires after the last seen duplicate.
//!
//!
//! ## Data structures vs. debouncers
//!
//! The debouncing logic itself is implemented as passive data structures in the
//! [buffer] module. A buffer only implements `.put` and `.get`, and
//! is independent of a particular way of organizing asynchronicity, be it
//! periodic polling, threads, async streams or some other magic.
//!
//! The crate also provides [threaded debouncers](thread) built on top of the
//! buffers, which is what you probably want to use most of the time.
//!
//!
//! ## Example usage
//!
//! ```rust
//! use debounce::EventDebouncer;
//! use std::thread::sleep;
//! use std::time::Duration;
//!
//! let delay = Duration::from_millis(10);
//! let debouncer = EventDebouncer::new(delay, move |data: String| {
//!     println!("{}", data);
//! });
//!
//! debouncer.put(String::from("foo"));
//! debouncer.put(String::from("foo"));
//! debouncer.put(String::from("bar"));
//! sleep(delay);
//!
//! // prints one "foo" and one "bar" after the delay
//! ```
//!
//! A working command-line debouncer is implemented in `examples/debounce.rs`
//! and can be used right away:
//!
//! ```sh
//! inotifywait -m -e close_write . | debounce -d 100
//! ```

pub mod buffer;
pub mod thread;

pub use thread::{EventDebouncer, MixedEventDebouncer};
