use std::path::PathBuf;

use figment::{
  providers::{Format, Toml},
  Figment
};

use winreg::{enums::HKEY_LOCAL_MACHINE, RegKey};

use crate::Error;


fn get_params_subkey(service_name: &str) -> Result<winreg::RegKey, Error> {
  Ok(
    RegKey::predef(HKEY_LOCAL_MACHINE)
      .open_subkey("SYSTEM\\CurrentControlSet\\Services")?
      .open_subkey(service_name)?
      .open_subkey("Parameters")?
  )
}


/// Load a DDMW application configuration file from a service's `Parameters`
/// subkey.  The path and filename of the configuration file should be stored
/// in a key called `AppConf`.
///
/// Unlike the [`conf::load()`](crate::conf::load) function there is no
/// fallback to getting config pathname from an environment variable or a
/// default location.  The `AppConf` value must exist and its data must point
/// to a valid configuration file.
pub fn load(service_name: &str) -> Result<Figment, Error> {
  let key = "AppConf";
  let subkey = get_params_subkey(service_name)?;
  let fname = subkey.get_value::<String, &str>(key)?;
  let fname = PathBuf::from(fname);
  let fig = Figment::new().merge(Toml::file(fname));

  Ok(fig)
}

// vim: set ft=rust et sw=2 ts=2 sts=2 cinoptions=2 tw=79 :
