use bytes::Bytes;
use criterion::{criterion_group, criterion_main, BenchmarkId, Criterion, Throughput};
use dbus_message_parser::message::Message;
use std::time::Duration;

static MSG_1: &[u8; 90] =
    b"\x6c\x02\x01\x01\x0a\x00\x00\x00\x01\x00\x00\x00\x3d\x00\x00\x00\x06\x01\x73\x00\x05\x00\x00\
    \x00\x3a\x31\x2e\x39\x38\x00\x00\x00\x05\x01\x75\x00\x01\x00\x00\x00\x08\x01\x67\x00\x01\x73\
    \x00\x00\x07\x01\x73\x00\x14\x00\x00\x00\x6f\x72\x67\x2e\x66\x72\x65\x65\x64\x65\x73\x6b\x74\
    \x6f\x70\x2e\x44\x42\x75\x73\x00\x00\x00\x00\x05\x00\x00\x00\x3a\x31\x2e\x39\x38\x00";

static MSG_2: &[u8; 308] =
    b"\x6c\x02\x01\x01\xec\x00\x00\x00\x60\x00\x00\x00\x36\x00\x00\x00\x06\x01\x73\x00\x06\x00\x00\
    \x00\x3a\x31\x2e\x31\x30\x35\x00\x00\x08\x01\x67\x00\x0a\x61\x7b\x73\x28\x62\x67\x61\x76\x29\
    \x7d\x00\x05\x01\x75\x00\x39\x01\x00\x00\x07\x01\x73\x00\x05\x00\x00\x00\x3a\x31\x2e\x39\x39\
    \x00\x00\x00\xe4\x00\x00\x00\x00\x00\x00\x00\x04\x00\x00\x00\x71\x75\x69\x74\x00\x00\x00\x00\
    \x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0c\x00\x00\
    \x00\x6e\x65\x77\x2d\x64\x6f\x63\x75\x6d\x65\x6e\x74\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\
    \x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0b\x00\x00\x00\x70\x72\x65\x66\x65\
    \x72\x65\x6e\x63\x65\x73\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\
    \x09\x00\x00\x00\x73\x68\x6f\x72\x74\x63\x75\x74\x73\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\
    \x00\x00\x00\x00\x00\x00\x00\x00\x00\x04\x00\x00\x00\x68\x65\x6c\x70\x00\x00\x00\x00\x00\x00\
    \x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05\x00\x00\x00\x61\
    \x62\x6f\x75\x74\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\
    \x00\x00\x00\x00\x0a\x00\x00\x00\x6e\x65\x77\x2d\x77\x69\x6e\x64\x6f\x77\x00\x00\x01\x00\x00\
    \x00\x00\x00\x00\x00\x00\x00\x00\x00";

static MSG_3: &[u8; 252] =
    b"\x6c\x04\x01\x01\x74\x00\x00\x00\xb8\x00\x00\x00\x76\x00\x00\x00\x01\x01\x6f\x00\x10\x00\x00\
    \x00\x2f\x6f\x72\x67\x2f\x67\x6e\x6f\x6d\x65\x2f\x64\x66\x65\x65\x74\x00\x00\x00\x00\x00\x00\
    \x00\x00\x02\x01\x73\x00\x0f\x00\x00\x00\x6f\x72\x67\x2e\x67\x74\x6b\x2e\x41\x63\x74\x69\x6f\
    \x6e\x73\x00\x08\x01\x67\x00\x16\x61\x73\x61\x7b\x73\x62\x7d\x61\x7b\x73\x76\x7d\x61\x7b\x73\
    \x28\x62\x67\x61\x76\x29\x7d\x00\x00\x00\x00\x00\x03\x01\x73\x00\x07\x00\x00\x00\x43\x68\x61\
    \x6e\x67\x65\x64\x00\x07\x01\x73\x00\x05\x00\x00\x00\x3a\x31\x2e\x38\x39\x00\x00\x00\x00\x00\
    \x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x5c\x00\x00\x00\x00\x00\x00\x00\x04\
    \x00\x00\x00\x68\x65\x6c\x70\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\
    \x00\x00\x00\x00\x00\x00\x00\x00\x05\x00\x00\x00\x61\x62\x6f\x75\x74\x00\x00\x00\x00\x00\x00\
    \x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x04\x00\x00\x00\x71\x75\
    \x69\x74\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00";

static MSG_4: &[u8; 160] =
    b"\x6c\x01\x00\x01\x00\x00\x00\x00\xbd\x00\x00\x00\x8e\x00\x00\x00\x01\x01\x6f\x00\x04\x00\x00\
    \x00\x2f\x6f\x72\x67\x00\x00\x00\x00\x02\x01\x73\x00\x23\x00\x00\x00\x6f\x72\x67\x2e\x66\x72\
    \x65\x65\x64\x65\x73\x6b\x74\x6f\x70\x2e\x44\x42\x75\x73\x2e\x49\x6e\x74\x72\x6f\x73\x70\x65\
    \x63\x74\x61\x62\x6c\x65\x00\x00\x00\x00\x00\x06\x01\x73\x00\x1c\x00\x00\x00\x6f\x72\x67\x2e\
    \x66\x72\x65\x65\x64\x65\x73\x6b\x74\x6f\x70\x2e\x46\x69\x6c\x65\x4d\x61\x6e\x61\x67\x65\x72\
    \x31\x00\x00\x00\x00\x03\x01\x73\x00\x0a\x00\x00\x00\x49\x6e\x74\x72\x6f\x73\x70\x65\x63\x74\
    \x00\x00\x00\x00\x00\x00\x07\x01\x73\x00\x05\x00\x00\x00\x3a\x31\x2e\x38\x39\x00\x00\x00";

static MESSAGES: [(&[u8], &str); 4] = [
    (MSG_1, "msg_1"),
    (MSG_2, "msg_2"),
    (MSG_3, "msg_3"),
    (MSG_4, "msg_4"),
];

fn decode(c: &mut Criterion) {
    let mut group = c.benchmark_group("decode");
    for (msg, name) in MESSAGES.iter() {
        let msg = Bytes::from_static(msg);
        let name = name.to_owned();
        let throughput = Throughput::Bytes(msg.len() as u64);
        let benchmark_id = BenchmarkId::new(name, msg.len());
        group.throughput(throughput);
        group.bench_with_input(benchmark_id, &msg, |b, msg| {
            b.iter(|| Message::decode(msg.clone()).unwrap())
        });
    }
    group.finish();
}

fn encode(c: &mut Criterion) {
    let mut group = c.benchmark_group("encode");
    for (msg, name) in MESSAGES.iter() {
        let msg = Bytes::from_static(msg);
        let (msg, _) = Message::decode(msg).unwrap();
        let bytes = msg.encode().unwrap();
        let name = name.to_owned();
        let throughput = Throughput::Bytes(bytes.len() as u64);
        let benchmark_id = BenchmarkId::new(name, bytes.len());
        group.throughput(throughput);
        group.bench_with_input(benchmark_id, &msg, |b, msg| {
            b.iter(|| msg.encode().unwrap())
        });
    }
    group.finish();
}

criterion_group! {
    name = benches;
    config = Criterion::default()
        .sample_size(1024)
        .measurement_time(Duration::from_secs(10));
    targets = decode, encode
}
criterion_main!(benches);
