//! Minidump generation with optional automatic naming.
//!
//! # Notes
//! - Microsoft advises against a process creating a dump of itself, which is
//!   understandable, but at the same time they suggest how to solve issues
//!   caused by doing precisely this.
//! - The thread calling `MiniDumpWriteDump()` will probably not be able to
//!   produce a usable callstack.  There are tricks to work around this, one of
//!   which is to call the function from a separate thread dedicated to
//!   creating the minidump.
//! - Some of the debugging functions are non-reentrant/single-threaded,
//!   including the `MiniDumpWriteDump()` function.  This means that the caller
//!   must ensure that two or more threads do not try to call it
//!   simultaneously.

use std::borrow::Borrow;
use std::path::{Path, PathBuf};

use crate::bindings::Windows::Win32::Debug::*;
use crate::bindings::Windows::Win32::FileSystem::*;
use crate::bindings::Windows::Win32::SystemServices::*;
use crate::bindings::Windows::Win32::WindowsProgramming::*;

use crate::err::Error;


/// Specify what type of minidump to write.
#[derive(Copy, Clone)]
pub enum DumpType {
  Mini,
  FullMem
}


/// MiniDump configuration buffer.
///
/// Configure type, location and name of minidumps generated by
/// [`create`].
#[derive(Clone)]
pub struct DumpInfo {
  /// The type of minidump to produce.
  pub mdtype: DumpType,

  /// Directory into which the dump will be created.  If not set, the dumps
  /// will be written to the current directory.
  ///
  /// If the current directory is not writable of it is not known which
  /// directory the calling process will be in when a minidump has been
  /// requested then it is recommended that `dumpsdir` is explicitly set to an
  /// absolute directory which exists and the process has write access to.
  pub dumpsdir: Option<PathBuf>,

  /// File name of the minidump.
  ///
  /// If this is `None` the library will attempt to generate a name by calling
  /// [`std::env::current_exe()`] and replacing the extension with `.dmp`.
  /// If the name of the current executable can not be determined, the
  /// default name `mini.dmp`.
  ///
  /// If `Some` value is set, this will be used as the minidump file name.
  pub name: Option<PathBuf>,

  /// If `true` the dump file names will be sequenced.  The naming convention
  /// is to remove the `.dmp` exptension, if it exists, and append
  /// `-<sequence>.dmp`, where `<sequence>` is a four digit hexadecimal
  /// number starting at `0000`.  If `false` the minidump file will be
  /// overwritten if it already exists.
  ///
  /// If the sequence number `ffff` has been reached it will be overwritten
  /// each run.
  ///
  /// Note: If required the file name will lossy converted to a string.  It is
  ///       recommended that the caller ensures that the name can be
  ///       represented as a unicode string.
  pub seq: bool
}

impl Default for DumpInfo {
  fn default() -> Self {
    DumpInfo {
      mdtype: DumpType::Mini,
      dumpsdir: None,
      name: None,
      seq: false
    }
  }
}


/// Create a minidump of the current process.
///
/// # Notes
/// - The calling thread's call stack will probably not be produced accurately
///   in the minidump.  A work-around is to call this function from a thread
///   (which is dedicated to calling this function).
pub fn create<D>(di: D) -> Result<(), Error>
where
  D: Borrow<DumpInfo>
{
  let di = di.borrow();

  let fname = gen_fname(di)?;

  // CreateFileW() doesn't support Path(Buf) :/
  let fname = match fname.to_str() {
    Some(nm) => nm,
    None => "mini.dmp"
  };

  //let file = fs::File::create(fname).unwrap();

  let hfile = unsafe {
    CreateFileW(
      fname,
      FILE_ACCESS_FLAGS::FILE_GENERIC_WRITE,
      FILE_SHARE_MODE::FILE_SHARE_READ,
      std::ptr::null_mut(),
      FILE_CREATION_DISPOSITION::CREATE_ALWAYS,
      FILE_FLAGS_AND_ATTRIBUTES::FILE_ATTRIBUTE_NORMAL,
      None
    )
  };
  if hfile.is_invalid() {
    //windows::HRESULT::from_thread().ok()?;
    let e = format!("Unable to open output file {}", fname);
    return Err(Error::MiniDump(e));
  }

  let hprocess = unsafe { GetCurrentProcess() };
  let pid = std::process::id();
  //let hfile = file.into_raw_handle();


  unsafe {
    MiniDumpWriteDump(
      hprocess,
      pid,
      hfile,
      match di.mdtype {
        DumpType::Mini => MINIDUMP_TYPE::MiniDumpNormal,
        DumpType::FullMem => MINIDUMP_TYPE::MiniDumpWithFullMemory
      },
      std::ptr::null_mut(),
      std::ptr::null_mut(),
      std::ptr::null_mut()
    )
  };

  unsafe { CloseHandle(hfile) };

  Ok(())
}


fn gen_fname(di: &DumpInfo) -> Result<PathBuf, Error> {
  // Determine the base name (i.e. name without .dmp extension).
  let basefname = match &di.name {
    Some(nm) => match nm.file_stem() {
      Some(s) => PathBuf::from(s),
      None => {
        return Err(Error::MiniDump("Invalid minidump file name".to_string()));
      }
    },
    None => match std::env::current_exe() {
      Ok(nm) => match nm.file_stem() {
        Some(s) => PathBuf::from(s),
        None => PathBuf::from("mini")
      },
      Err(_) => PathBuf::from("mini")
    }
  };

  let fname = if di.seq {
    let mut i = 0;
    loop {
      let nm = format!("{}-{:04x}", basefname.display(), i);
      let nm = Path::new(&nm);
      let fullname = if let Some(dir) = &di.dumpsdir {
        dir.join(nm.with_extension("dmp"))
      } else {
        nm.with_extension("dmp")
      };
      if fullname.exists() {
        if i < 0xffff {
          i += 1;
          continue;
        }
      }
      break fullname;
    }
  } else {
    if let Some(dir) = &di.dumpsdir {
      dir.join(basefname.with_extension("dmp"))
    } else {
      basefname.with_extension("dmp")
    }
  };

  Ok(fname)
}

#[cfg(test)]
mod tests {
  use super::*;

  #[test]
  fn hardcoded_filename() {
    let di = DumpInfo {
      name: Some(PathBuf::from("foo")),
      ..Default::default()
    };

    assert_eq!(gen_fname(&di).unwrap(), PathBuf::from("foo.dmp"));
  }


  #[test]
  fn hardcoded_pathname() {
    let di = DumpInfo {
      dumpsdir: Some(PathBuf::from("C:\\Temp")),
      name: Some(PathBuf::from("foo")),
      ..Default::default()
    };

    assert_eq!(gen_fname(&di).unwrap(), PathBuf::from("C:\\Temp\\foo.dmp"));
  }


  #[test]
  fn seq_pathname() {
    // ToDo: Remove existing foo-xxxx.dmp files, then create them and make sure
    //       new ones are generated.
    let di = DumpInfo {
      dumpsdir: Some(PathBuf::from("C:\\Temp")),
      name: Some(PathBuf::from("foo")),
      seq: true,
      ..Default::default()
    };

    assert_eq!(
      gen_fname(&di).unwrap(),
      PathBuf::from("C:\\Temp\\foo-0000.dmp")
    );
  }
}

// vim: set ft=rust et sw=2 ts=2 sts=2 cinoptions=2 tw=79 :
