use super::Dumper;
use crate::args::Options;
use crate::writer;
use crate::writer::Writer;
use mysql;
use mysql::prelude::*;
use std::process::exit;
use std::sync::mpsc::sync_channel;
use std::thread;

pub struct Mysql;

impl Dumper for Mysql {
    fn dump(&self, options: &Options) {
        let opts = mysql::OptsBuilder::new()
            .ip_or_hostname(Some(&options.host))
            .tcp_port(options.port)
            .user(Some(&options.username))
            .pass(Some(&options.password))
            .db_name(Some(&options.database));
        let mut conn = mysql::Conn::new(opts).expect("Connect to mysql server error");

        let mut writer: Box<dyn Writer + Send + Sync> = match options.format.as_str() {
            "csv" => writer::csv::Csv::new(&options.output),
            "xls" => writer::xls::Xls::new(&options.output),
            _ => return,
        };

        let result_set: Vec<(String,)> = conn
            .query(format!(
                "select column_name from information_schema.columns where table_name = '{}' and table_schema = '{}'",
                options.table, options.database
            ))
            .expect("Query mysql error");
        let name_row: Vec<String> = result_set.into_iter().map(|row| row.0).collect();
        writer.write_row(name_row.clone());

        let mut sql = "select ".to_owned();
        sql += &name_row.join(",");

        let (sender, receiver) = sync_channel(128);
        let writer_thread = thread::spawn(move || {
            receiver.iter().for_each(|row| writer.write_row(row));
        });

        conn.query_iter(format!(
            "{} from {} limit {} offset {}",
            sql, options.table, options.rows, options.start
        ))
        .unwrap()
        .for_each(|row| match row {
            Ok(mut row) => {
                let mut data_row: Vec<String> = vec![];
                for i in 0..row.len() {
                    data_row.push(row.take(i).unwrap());
                }
                sender
                    .send(data_row)
                    .expect("Send data row to channel error");
            }
            Err(err) => {
                println!("Query mysql error: {:?}", err);
                exit(1);
            }
        });

        drop(sender);
        writer_thread
            .join()
            .expect("The writer thread has panicked");
        println!("finished!");
    }
}
