use std::sync::mpsc::sync_channel;
use std::thread;

use super::Dumper;
use crate::writer;
use crate::writer::Writer;

use hex::ToHex;
use sqlite;

pub struct Sqlite;

impl Dumper for Sqlite {
    fn dump(&self, options: &crate::args::Options) {
        let conn = sqlite::open(&options.database).expect("Connect to sqlite error");
        let mut writer: Box<dyn Writer + Send + Sync> = match options.format.as_str() {
            "csv" => writer::csv::Csv::new(&options.output),
            "xls" => writer::xls::Xls::new(&options.output),
            _ => return,
        };

        let mut statement = conn
            .prepare(format!("PRAGMA table_info('{}')", options.table))
            .unwrap();

        let mut name_row: Vec<String> = vec![];
        loop {
            match statement.next() {
                Ok(sqlite::State::Row) => {
                    name_row.push(
                        statement
                            .read::<String>(1)
                            .expect("Read name column from sqlite error"),
                    );
                }
                Ok(sqlite::State::Done) => break,
                Err(err) => {
                    println!("Advance sqlite statement error: {:?}", err);
                    return;
                }
            }
        }

        assert_ne!(name_row.len(), 0);
        writer.write_row(name_row.clone());

        let mut sql = "select ".to_owned();
        sql += &name_row.join(",");

        let (sender, receiver) = sync_channel(128);
        let writer_thread = thread::spawn(move || {
            receiver.iter().for_each(|row| writer.write_row(row));
        });

        let mut cursor = conn
            .prepare(format!(
                "{} from {} limit {} offset {}",
                sql, options.table, options.rows, options.start
            ))
            .unwrap()
            .into_cursor();

        loop {
            match cursor.next() {
                Ok(Some(row)) => {
                    let data_row: Vec<String> = row
                        .iter()
                        .map(|column| match column.kind() {
                            sqlite::Type::Binary => {
                                column.as_binary().unwrap().encode_hex::<String>()
                            }
                            sqlite::Type::Float => column.as_float().unwrap().to_string(),
                            sqlite::Type::Integer => column.as_integer().unwrap().to_string(),
                            sqlite::Type::String => column.as_string().unwrap().to_owned(),
                            sqlite::Type::Null => "".to_owned(),
                        })
                        .collect();
                    sender
                        .send(data_row)
                        .expect("Send data row to channel error");
                }
                Ok(None) => break,
                Err(err) => {
                    println!("Advance sqlite cursor error: {:?}", err);
                    return;
                }
            }
        }

        drop(sender);
        writer_thread
            .join()
            .expect("The writer thread has panicked");
        println!("finished!");
    }
}
