use super::Dumper;
use crate::args::Options;
use crate::writer;
use crate::writer::Writer;
use postgres::tls::NoTls;
use postgres::Client;
use postgres::SimpleQueryMessage::{CommandComplete, Row};
use std::sync::mpsc::sync_channel;
use std::thread;

pub struct Postgresql;

impl Dumper for Postgresql {
    fn dump(&self, options: &Options) {
        let mut client = Client::connect(
            &format!(
                "postgresql://{}:{}@{}:{}/{}",
                options.username, options.password, options.host, options.port, options.database
            ),
            NoTls,
        )
        .expect("Connect to postgresql server error");

        let mut writer: Box<dyn Writer + Send + Sync> = match options.format.as_str() {
            "csv" => writer::csv::Csv::new(&options.output),
            "xls" => writer::xls::Xls::new(&options.output),
            _ => return,
        };

        let row = client
            .query_one(
                "select column_name from information_schema.columns where table_name = $1",
                &[&options.table],
            )
            .expect("Query postgresql error");

        let mut name_row: Vec<String> = vec![];
        for i in 0..row.len() {
            name_row.push(row.get(i));
        }
        writer.write_row(name_row.clone());

        let mut sql = "select ".to_owned();
        sql += &name_row.join(",");
        let sql = format!(
            "{} from {} limit {} offset {}",
            sql, options.table, options.rows, options.start
        );
        let query = format!("DECLARE \"dbdump-postgresql-cursor\" CURSOR FOR {}", sql);

        client.execute("BEGIN", &[]).unwrap();
        client.execute(&query, &[]).unwrap();

        let (sender, receiver) = sync_channel(128);
        let writer_thread = thread::spawn(move || {
            receiver.iter().for_each(|row| writer.write_row(row));
        });

        let fetch_query = "FETCH 10 FROM \"dbdump-postgresql-cursor\"";
        loop {
            let messages = client
                .simple_query(fetch_query)
                .expect("Query postgresql error");
            for message in &messages {
                match message {
                    Row(row) => {
                        let mut data_row: Vec<String> = vec![];
                        for i in 0..row.len() {
                            data_row.push(row.get(i).unwrap().to_owned());
                        }
                        sender
                            .send(data_row)
                            .expect("Send data row to channel error");
                    }
                    CommandComplete(_) => (),
                    _ => (),
                }
            }

            if messages.len() < 10 {
                client
                    .execute("CLOSE \"dbdump-postgresql-cursor\"", &[])
                    .unwrap();
                client.execute("COMMIT", &[]).unwrap();
                break;
            }
        }

        drop(sender);
        writer_thread
            .join()
            .expect("The writer thread has panicked");
        println!("finished!");
    }
}
