//
// Copyright © 2022, Oleg Lelenkov
// License: BSD 3-Clause
// Authors: Oleg Lelenkov
//

use tracing::subscriber::Interest;
use tracing::{span, Event, Metadata};
use tracing_subscriber::layer::{Context, Layer};
use tracing_subscriber::registry::Registry;

use super::DynLayer;

pub struct CombinedLayer {
    layers: Vec<DynLayer>,
}

impl CombinedLayer {
    pub fn new(layers: Vec<DynLayer>) -> Self {
        Self { layers }
    }
}

impl Layer<Registry> for CombinedLayer {
    fn register_callsite(&self, metadata: &'static Metadata<'static>) -> Interest {
        for layer in self.layers.iter() {
            layer.register_callsite(metadata);
        }
        Interest::always()
    }

    fn on_layer(&mut self, subscriber: &mut Registry) {
        for layer in self.layers.iter_mut() {
            layer.on_layer(subscriber);
        }
    }

    fn enabled(&self, _meta: &Metadata, _ctx: Context<Registry>) -> bool {
        !self.layers.is_empty()
    }

    fn on_new_span(&self, attrs: &span::Attributes, id: &span::Id, ctx: Context<Registry>) {
        for layer in self.layers.iter() {
            layer.on_new_span(attrs, id, ctx.clone());
        }
    }

    fn on_event(&self, event: &Event, ctx: Context<Registry>) {
        for layer in self.layers.iter() {
            layer.on_event(event, ctx.clone());
        }
    }

    fn on_enter(&self, id: &span::Id, ctx: Context<Registry>) {
        for layer in self.layers.iter() {
            layer.on_enter(id, ctx.clone());
        }
    }

    fn on_exit(&self, id: &span::Id, ctx: Context<Registry>) {
        for layer in self.layers.iter() {
            layer.on_exit(id, ctx.clone());
        }
    }

    fn on_record(&self, id: &span::Id, values: &span::Record, ctx: Context<Registry>) {
        for layer in self.layers.iter() {
            layer.on_record(id, values, ctx.clone());
        }
    }

    fn on_follows_from(&self, id: &span::Id, follows: &span::Id, ctx: Context<Registry>) {
        for layer in self.layers.iter() {
            layer.on_follows_from(id, follows, ctx.clone());
        }
    }

    fn on_id_change(&self, old: &span::Id, new: &span::Id, ctx: Context<Registry>) {
        for layer in self.layers.iter() {
            layer.on_id_change(old, new, ctx.clone());
        }
    }

    fn on_close(&self, id: span::Id, ctx: Context<Registry>) {
        for layer in self.layers.iter() {
            layer.on_close(id.clone(), ctx.clone());
        }
    }
}
