use chrono::Utc;

use reqwest::StatusCode;

use serde::Serialize;

use snafu::{ResultExt, Snafu};

use dalloriam_cloud_protocol::hose::GenericMessage;

use crate::{service, Credentials};

#[derive(Debug, Snafu)]
pub enum HoseError {
    AccessDenied,
    DispatchError { source: reqwest::Error },
    UnexpectedError { status: StatusCode },
}

type Result<T> = std::result::Result<T, HoseError>;

pub struct HoseClient {
    client: reqwest::Client,
    credentials: Credentials,
}

impl HoseClient {
    pub fn new(credentials: Credentials) -> Self {
        Self {
            client: reqwest::Client::default(),
            credentials,
        }
    }

    pub async fn dispatch<T: Serialize>(&self, key: &str, body: &T) -> Result<()> {
        let msg = GenericMessage {
            key,
            body,
            time: Utc::now(),
        };

        let resp = self
            .client
            .post(&format!("{}/dispatch", service::HOSE_HOST))
            .header("Authorization", self.credentials.to_string())
            .json(&msg)
            .send()
            .await
            .context(DispatchSnafu)?;

        if !resp.status().is_success() {
            return match resp.status() {
                StatusCode::UNAUTHORIZED | StatusCode::FORBIDDEN => Err(HoseError::AccessDenied),
                _ => Err(HoseError::UnexpectedError {
                    status: resp.status(),
                }),
            };
        }

        Ok(())
    }
}
