use seahash::hash;

const MASKS: [u64; 65] = [
    0,
    1,
    3,
    7,
    15,
    31,
    63,
    127,
    255,
    511,
    1023,
    2047,
    4095,
    8191,
    16383,
    32767,
    65535,
    131071,
    262143,
    524287,
    1048575,
    2097151,
    4194303,
    8388607,
    16777215,
    33554431,
    67108863,
    134217727,
    268435455,
    536870911,
    1073741823,
    2147483647,
    4294967295,
    8589934591,
    17179869183,
    34359738367,
    68719476735,
    137438953471,
    274877906943,
    549755813887,
    1099511627775,
    2199023255551,
    4398046511103,
    8796093022207,
    17592186044415,
    35184372088831,
    70368744177663,
    140737488355327,
    281474976710655,
    562949953421311,
    1125899906842623,
    2251799813685247,
    4503599627370495,
    9007199254740991,
    18014398509481983,
    36028797018963967,
    72057594037927935,
    144115188075855871,
    288230376151711743,
    576460752303423487,
    1152921504606846975,
    2305843009213693951,
    4611686018427387903,
    9223372036854775807,
    18446744073709551615,
];

const ALT_HASH: [u64; 256] = [
    8706543430870418941,
    5841279983151608680,
    6215326323105245339,
    10485622740458680607,
    11107982111527004974,
    441767137009022216,
    4810949641127369933,
    9185462851015206250,
    11380200210756233104,
    4192023372858266261,
    6448469143720294743,
    8917388347039983681,
    12334988489921557722,
    9195634891349698153,
    11584156053403770360,
    17479415582321224937,
    6651029026926081805,
    9540066605718248802,
    4506971699369181381,
    10922736322397478392,
    18408433360944122726,
    5113817322932416322,
    10192663111597099709,
    12359136565904111959,
    14371410548495126979,
    12786522659195975263,
    12155282352181316174,
    13329477317560214841,
    12272807196886509598,
    225868468655136518,
    283879565895082989,
    245091375483304822,
    14142344341502767213,
    7639389811471966827,
    8640838183246683570,
    14262887129989736451,
    4683701774435886119,
    4255030481804771146,
    3721812351796781109,
    15548215111414605664,
    16710301143469405233,
    10812149864820974956,
    7075416800807765110,
    15201392300412275249,
    9245427870045768798,
    7111326055513658739,
    13518565203113096007,
    4888583578966393327,
    16698015096624893553,
    2929670310044459792,
    17529823629616385475,
    6951613828128883812,
    9943592476348511138,
    9592814376127833256,
    549805554402871179,
    12734621804655080495,
    17935685719023588999,
    15450621252045431229,
    3188107047132796823,
    7958588281490684230,
    13835972819577822788,
    17782105574167630658,
    722976464608453500,
    16612931893869889191,
    2785067955193975050,
    11008204808610869189,
    3965468562767638881,
    8895096683356775633,
    17657332256906722271,
    9826675979335290647,
    8046760446598255930,
    10846501041380513582,
    8268467241425705344,
    7820685027799762759,
    9968483728905060652,
    6629847097473447578,
    4503448431571363738,
    12570721396852748618,
    213964659381542709,
    9508027552405936468,
    7570496784350298076,
    266591330903080136,
    8912304157236285200,
    484616793778371061,
    9416369421163358073,
    12800196476891370983,
    9130993494598814669,
    12041017042497249598,
    9194540312741985656,
    16138774767392728047,
    10795286308907093950,
    14698677735298450564,
    1650794019137419762,
    16789359497387250791,
    16378924276709600671,
    1922289906489455871,
    4872967784508945678,
    3009532316786026829,
    8907395914187219719,
    18326680962986713139,
    16284457459055993821,
    11957547300847398003,
    8060362414078111917,
    17304668238460453134,
    1004630068229131464,
    15021502172862101004,
    1114819617094146642,
    2580355813004382039,
    9846629712022200445,
    4666386448732604556,
    9745224974844912069,
    10656420646993080621,
    8610366768398575394,
    209905813532434341,
    6362955193336106334,
    3222580457287756091,
    17044711358146693026,
    4375449034917630022,
    7585918148212236266,
    6283025383019450298,
    13259461164541992611,
    15593781351455431223,
    10711956318185209753,
    1438821901844615515,
    15528452422014509147,
    5052460273704478018,
    16263655947820251196,
    8148443451270673248,
    7119222946422613310,
    6413718312817935097,
    13450485197710023174,
    15820759858101192144,
    15992756818933446570,
    9329267768936995843,
    10470675471724420643,
    10980838142357162788,
    17355117362412287022,
    1259568893613927165,
    1018308218084368684,
    4226701599749572446,
    16257636105976563860,
    6705470177628182016,
    11642947443155004788,
    9766665717460925820,
    16030511881650827278,
    18220255696163033901,
    245580789939364594,
    12353639007163351197,
    10401936625861770967,
    16093085842180152119,
    17919562608837130375,
    4240105080348451917,
    7538772458920233066,
    7107797428248387352,
    10626995774494508084,
    11646246798115436779,
    3291279567904201213,
    10698879415374990494,
    14799986012370376611,
    14633844252921129358,
    11354044316300679470,
    11932869691484732927,
    8170189431019825721,
    11192112391529588142,
    6698402682294275138,
    3304481951316181700,
    14000894325860251974,
    6458368906063596943,
    11791734883044879628,
    4363062015814434231,
    6096753730927982222,
    15638226648481409074,
    9856166997429400700,
    1563881106706340881,
    11221106407003697790,
    14746259275098111086,
    14985116290126017309,
    13532244187402333296,
    8885066748276037314,
    2532970192449129871,
    844291328304588163,
    10744746894569889886,
    14502783522197635894,
    16862976771563183441,
    13078169881214682352,
    8155653542088168455,
    2270807018703666934,
    730904892379690354,
    10703968767027087835,
    13857526616163989533,
    3277575953836679493,
    14407376690648407577,
    4226128901775540809,
    9165760600366237724,
    10927365416129918118,
    10412576175383873010,
    7785768021343161364,
    17715881364603619977,
    10034376467670767868,
    3952410003848476588,
    14233000679133642590,
    6299681824378301250,
    17227400206377150916,
    11881689285474545557,
    6957649870958667863,
    2626335199972001971,
    6158590382225266343,
    1080329125805451604,
    7058063538070572977,
    15004058616457175951,
    16154507325316593208,
    10037054838184078184,
    14520356353718481805,
    18291191664738060797,
    4335337003338060315,
    7798066410472417318,
    9831222661552430905,
    10941656208595521213,
    3610450067886516743,
    14098720450197273427,
    9154535105663855483,
    17730458950619767424,
    3478929070597686746,
    267128081773890932,
    14661952438273390280,
    592190050035571971,
    14390402862226483614,
    7498522598573179691,
    8114887307153265163,
    804602643653796829,
    7813631357943616924,
    10420668047098104672,
    3658946373227250015,
    5188093849880705649,
    12985911135289494005,
    16807618691360259796,
    7406421251572400745,
    2245274194566413452,
    69463990612319238,
    12229264837875354372,
    15824221098692839002,
    8517226763975203379,
    14111006860217444247,
    5439777640158342268,
    9769116601198101931,
    13411023565239239314,
    14461675873906545352,
    6151360885816763151,
    7177563267458015364,
    13180930152834317168,
    17636806686541702274,
    16120014739386934181,
    457481492959674793,
    17233023865740887841,
    11080911332779745796,
    7729016964490111755,
];

pub fn upper_power2(x: u64) -> u64 {
    let mut m = x - 1;
    m |= m >> 1;
    m |= m >> 2;
    m |= m >> 4;
    m |= m >> 8;
    m |= m >> 16;
    m |= m >> 32;
    m + 1
}
//
//pub fn get_index_hash(u: u64) -> u64 {
//    u & (table_ -> NumBuckets() - 1)
//}
//

pub struct FingerIndex {
    pub fp: u8,
    pub i1: u64,
    pub i2: u64,
}

pub fn get_alt_index(fp: u8, i: u64, pow: usize) -> u64 {
    let mask = MASKS[pow];
    let hash = ALT_HASH[fp as usize] & mask;
    (i & mask) ^ hash
}

pub fn get_fingerprint(data: &[u8]) -> u8 {
    (hash(data) % 255 + 1) as u8
}

pub fn get_indices_and_fingerprint(data: &[u8], pow: usize) -> FingerIndex {
    let index_hash = hash(data);
    let fp = get_fingerprint(data);
    let i1 = index_hash & MASKS[pow];
    let i2 = get_alt_index(fp, i1, pow);
    FingerIndex { fp, i1, i2 }
}

#[cfg(test)]
mod tests {
    use crate::util::upper_power2;

    #[test]
    fn test_upper_power2() {
        assert_eq!(upper_power2(1), 1);
        assert_eq!(upper_power2((1 << 4) - 1), 16);
        assert_eq!(upper_power2(1 << 10), 1024);
        assert_eq!(upper_power2((1 << 10) + 1), 2048);
    }
}
