pub fn strip_dag(original_dag: &str, vertice_to_strip: &str) -> String {
    let dag = original_dag.split(",").collect::<Vec<&str>>();

    // Allocating the patterns to strip given a character
    let strip_pattern_1 = format!("-{}", vertice_to_strip);
    let strip_pattern_2 = format!("{}-", vertice_to_strip);
    let strip_pattern_3 = format!("{}", vertice_to_strip);

    // Allocating a mutable vec to create the newly formatted dag. We know the length won't exceed the original's
    let mut new_dag: Vec<String> = Vec::with_capacity(dag.len());

    // The index that will allow us to peek within iterations to know if we're able to create a branch from a single vertice
    let mut next_idx = 0;

    for (idx, vertex) in dag.iter().enumerate() {
        if idx < next_idx {
            continue;
        }

        if vertex.contains(vertice_to_strip) {
            let stripped_vertex =
                strip_vertice(&strip_pattern_1, &strip_pattern_2, &strip_pattern_3, vertex);
            next_idx = idx + 1;

            // Check if we're able to create a branch
            while let Some(next_vertex) = check_next_vertex(
                &dag,
                next_idx,
                stripped_vertex.len(),
                vertice_to_strip,
                &stripped_vertex,
                &strip_pattern_1,
                &strip_pattern_2,
                &strip_pattern_3,
            ) {
                push_if_not_duplicate(next_vertex, &mut new_dag, idx);
                next_idx += 1;
            }
            continue;
        }

        // Nothing special to do on this vertice, and it wasn't used to create a branch. We just push it
        push_if_not_duplicate(String::from(*vertex), &mut new_dag, idx)
    }

    new_dag.sort();

    new_dag.join(",")
}

fn push_if_not_duplicate(current_vertex: String, new_dag: &mut Vec<String>, idx: usize) -> () {
    if idx != 0 {
        if let Some(vertex) = new_dag.get(idx - 1) {
            match (vertex.chars().last(), current_vertex.chars().last()) {
                // The last char of the previous vertex is equal to the last char of the current vertex (eg : "a-b,b" or "a-b,b-b"). We don't push
                (Some(vertice1), Some(vertice2)) if vertice1 == vertice2 => (),
                _ => new_dag.push(current_vertex),
            }
        }
    } else {
        // If idx is 0, there can't be any potential duplicate behind, we can just push
        new_dag.push(current_vertex);
    }
}

fn strip_vertice(
    strip_pattern_1: &String,
    strip_pattern_2: &String,
    strip_pattern_3: &String,
    vertex: &str,
) -> String {
    vertex
        .replace(strip_pattern_1, "")
        .replace(strip_pattern_2, "")
        .replace(strip_pattern_3, "")
}

fn check_next_vertex(
    dag: &Vec<&str>,
    next_idx: usize,
    vertex_len: usize,
    vertice_to_strip: &str,
    previous_stripped_vertex: &String,
    strip_pattern_1: &String,
    strip_pattern_2: &String,
    strip_pattern_3: &String,
) -> Option<String> {
    if vertex_len > 1 {
        // We've already got a branch
        None
    } else {
        match dag.get(next_idx) {
            // We have an unconnected vertice, and the next vertice will also be unconnected once its character to strip is removed. We create a branch from the two
            Some(next_vertex) if next_vertex.contains(vertice_to_strip) => Some(format!(
                "{}-{}",
                previous_stripped_vertex,
                strip_vertice(
                    strip_pattern_1,
                    strip_pattern_2,
                    strip_pattern_3,
                    next_vertex,
                )
            )),
            _ => None,
        }
    }
}
