/*!
  The primary table implementation
*/

use crate::header::Header;
use d4_framefile::mode::{ReadOnly, ReadWrite};
use d4_framefile::Directory;
use std::fs::File;
use std::io::Result;

mod uncompressed;

/// The result of decoding a value from a primary table
pub enum DecodeResult {
    /// Value at this location is definitely the value returned
    Definitely(i32),
    /// The value may be the value return, but the query to the secondary table is required
    Maybe(i32),
}

/// The trait that is used to write a primary table
pub trait PTableWriter: Sized {
    /// The writer type for a parallel chunk
    type Partition: PTablePartitionWriter;
    /// Create the primary table in the file
    fn create(directory: &mut Directory<'static, ReadWrite, File>, header: &Header)
        -> Result<Self>;
    /// Split the primary table into parallel partitions
    fn split(&mut self, header: &Header, size_limit: Option<usize>)
        -> Result<Vec<Self::Partition>>;
}

/// The trait that is used as one of the parallel partition split from the primary table writer
pub trait PTablePartitionWriter: Send {
    /// The type describes how we encode
    type EncoderType: Encoder;
    /// Create encoder for current partition
    fn as_encoder(&mut self) -> Self::EncoderType;
    /// Report the genome range this partition is responsible to
    fn region(&self) -> (&str, u32, u32);
    /// Report if the primary table can encode the value
    fn can_encode(&self, value: i32) -> bool;
    /// Report the bit width for this primary table
    fn bit_width(&self) -> usize;
}

/// Any type used to encode a value in primary table
pub trait Encoder {
    /// Encode a value at given location
    fn encode(&mut self, pos: usize, value: i32) -> bool;
}

/// Type that reads a D4 primary table
pub trait PTableReader: Sized {
    /// The type for parallel reading one of the partition
    type Partition: PTablePartitionReader;
    /// Create the reader instance
    fn create(directory: &mut Directory<'static, ReadOnly, File>, header: &Header) -> Result<Self>;
    /// Split the reader to parallel chunks
    fn split(&mut self, header: &Header, size_limit: Option<usize>)
        -> Result<Vec<Self::Partition>>;
}

/// The type that decodes one part of the primary table in paralell
pub trait PTablePartitionReader: Send {
    /// The decoder type
    type DecoderType: Decoder;
    /// Create decoder for current chunk
    fn as_decoder(&mut self) -> Self::DecoderType;
    /// Report the resposbile region
    fn region(&self) -> (&str, u32, u32);
    /// Report the bit width
    fn bit_width(&self) -> usize;
}

/// Any type that used for decoding the primary table
pub trait Decoder {
    /// Decode the value at one location
    fn decode(&mut self, pos: usize) -> DecodeResult;
    /// Decode a block of values - This is overriden for performance reasons
    fn decode_block<F: FnMut(usize, DecodeResult)>(
        &mut self,
        pos: usize,
        count: usize,
        mut handle: F,
    ) {
        for idx in 0..count {
            handle(pos + idx, self.decode(pos + idx));
        }
    }
}

/// The writer for uncompressed bit-array backed primary table
pub type UncompressedWriter = uncompressed::PrimaryTable<uncompressed::Writer>;
/// The reader for uncompressed bit-array backed primary table
pub type UncompressedReader = uncompressed::PrimaryTable<uncompressed::Reader>;
/// The partition writer for uncompressed bit-array backed primary table
pub type UncompressedPartWriter = uncompressed::PartialPrimaryTable<uncompressed::Writer>;
/// The partition reader for uncompressed bit-array backed primary table
pub type UncompressedPartReader = uncompressed::PartialPrimaryTable<uncompressed::Reader>;

/// The dceoder for bit-arrary primary table
pub type UncompressedDecoder = uncompressed::PrimaryTableCodec<uncompressed::Reader>;
/// The encoder for bit-arrray primary table
pub type UncompressedEncoder = uncompressed::PrimaryTableCodec<uncompressed::Writer>;
