use criterion::{black_box, criterion_group, criterion_main, Criterion};

use std::{
    fs::{read, File},
    io::Read,
};

fn criterion_benchmark(c: &mut Criterion) {
    c.bench_function("with_serde", |b| {
        let op = data::build_query();
        let menu_data = read("../menu.json").unwrap();

        b.iter(|| {
            let value_op = serde_json::from_slice(&menu_data).unwrap();
            op.decode_response(value_op).unwrap();
        });
    });

    c.bench_function("just_serde", |b| {
        let op = data::build_query();
        let menu_data = read("../menu.json").unwrap();

        b.iter(|| {
            serde_json::from_slice::<cynic::GraphQlResponse<serde_json::Value>>(&menu_data)
                .unwrap();
        });
    });

    c.bench_function("query string building", |b| {
        b.iter(|| {
            let op = data::build_query();
        });
    });
}

criterion_group!(benches, criterion_benchmark);
criterion_main!(benches);

mod data {
    pub use queries::*;

    pub fn build_query() -> cynic::Operation<'static, MenuQuery> {
        use cynic::QueryBuilder;

        MenuQuery::build(())
    }

    /// We're using cynic to build up our GraphQL query.  It uses a bunch of derive
    /// macros to take rust structs and convert them into the equivalent graphql query.
    ///
    /// These derive macros _can_ stand alone, but it saves some typing if you put them
    /// in a query_module like this, as then you don't need to provide the schema_path
    /// & query_module on every struct.
    #[cynic::schema_for_derives(file = "../schemas/menu.graphql")]
    pub(crate) mod queries {
        use super::schema;

        #[derive(cynic::QueryFragment, Debug)]
        #[cynic(graphql_type = "Query")]
        pub struct MenuQuery {
            pub menu: Menu,
        }

        #[derive(cynic::QueryFragment, Debug)]
        pub struct Menu {
            pub id: cynic::Id,
            pub name: String,
            pub sections: Vec<Section>,
        }

        #[derive(cynic::QueryFragment, Debug)]
        pub struct Section {
            pub items: Vec<MenuItem>,
        }

        #[derive(cynic::QueryFragment, Debug)]
        pub struct MenuItem {
            pub id: cynic::Id,
            pub slug: String,
            pub display_name: String,
            pub modifiers: Vec<Modifier>,
            pub fulfilment_types: Vec<FulfilmentType>,
            pub price: Option<Money>,
            pub eligible_rewards: Option<Vec<Reward>>,
            pub nutritional_info: NutritionalInfo,
        }

        #[derive(cynic::QueryFragment, Debug)]
        pub struct NutritionalInfo {
            pub facts: Option<NutritionalFacts>,
        }

        #[derive(cynic::QueryFragment, Debug)]
        pub struct NutritionalFacts {
            pub alcohol_by_volume_percent: Option<f64>,
        }

        #[derive(cynic::QueryFragment, Clone, Debug)]
        pub struct Reward {
            pub reward_type: RewardType,
            pub price_adjustment: Money,
        }

        #[derive(cynic::Enum, Clone, Copy, Debug)]
        pub enum RewardType {
            Green,
            Orange,
            Red,
        }

        #[derive(cynic::QueryFragment, Debug, Clone, Copy)]
        pub struct Money {
            pub currency_code: CurrencyCode,
            pub value: i32,
        }

        #[derive(cynic::QueryFragment, Debug)]
        pub struct Modifier {
            pub id: cynic::Id,
            pub slug: String,
            pub prompt_label: Option<String>,
            pub selectable_quantity: Option<i32>,
            pub show_if: Option<cynic::Id>,
            pub options: Vec<ModifierOption>,
            pub fulfilment_types: Vec<FulfilmentType>,
        }

        #[derive(cynic::QueryFragment, Debug)]
        pub struct ModifierOption {
            pub id: cynic::Id,
            pub slug: String,
            pub price: Money,
            pub display_name: String,
            pub fulfilment_types: Vec<FulfilmentType>,
            pub eligible_rewards: Vec<Reward>,
        }

        #[derive(cynic::Enum, Clone, Copy, Debug)]
        pub enum Market {
            UK,
            IE,
        }

        #[derive(cynic::Enum, Clone, Copy, Debug, PartialEq)]
        pub enum CurrencyCode {
            GBP,
            EUR,
        }

        #[derive(cynic::Enum, Clone, Copy, Debug)]
        pub enum FulfilmentType {
            Delivery,
            Collection,
            EatIn,
            AtTable,
        }
    }

    /// Cynic's derive macros require a schema module that is generated from the
    /// GraphQL schema.
    mod schema {
        cynic::use_schema!("../schemas/menu.graphql");
    }
}
