use schemars::JsonSchema;
use serde::{Deserialize, Serialize};

use cosmwasm_std::{Binary, Uint128};
use cw20::{Expiration, Logo};

#[derive(Serialize, Deserialize, Clone, Debug, PartialEq, JsonSchema)]
#[serde(rename_all = "snake_case")]
pub struct InstantiateMsg {
    pub cw20: cw20_base::msg::InstantiateMsg,
    pub token_factory_addr: String
}

#[derive(Serialize, Deserialize, Clone, Debug, PartialEq, JsonSchema)]
#[serde(rename_all = "snake_case")]
pub enum ExecuteMsg {
    Mint { recipient: String, amount: Uint128 },

    /* Implements CW20. Transfer is a base message to move tokens to another account without triggering actions*/
    Transfer { recipient: String, amount: Uint128 },

    /* Burn is a message to destroy tokens forever*/
    BurnFrom { owner: String, amount: Uint128 },
    
    /* Implements CW20. Send is a base message to transfer tokens to a contract and trigger an action
    on the receiving contract.*/
    Send {
        contract: String,
        amount: Uint128,
        msg: Binary,
    },
    /* Implements CW20 "approval" extension. Allows spender to access an additional amount tokens
    from the owner's (env.sender) account. If expires is Some(), overwrites current allowance
    expiration with this one. */
    IncreaseAllowance {
        spender: String,
        amount: Uint128,
        expires: Option<Expiration>,
    },
    /* Implements CW20 "approval" extension. Lowers the spender's access of tokens
    from the owner's (env.sender) account by amount. If expires is Some(), overwrites current
    allowance expiration with this one. */
    DecreaseAllowance {
        spender: String,
        amount: Uint128,
        expires: Option<Expiration>,
    },
    /* Implements CW20 "approval" extension. Transfers amount tokens from owner -> recipient
    if `env.sender` has sufficient pre-approval. */
    TransferFrom {
        owner: String,
        recipient: String,
        amount: Uint128,
    },
    /* Implements CW20 "approval" extension. Sends amount tokens from owner -> contract
    if `env.sender` has sufficient pre-approval. */
    SendFrom {
        owner: String,
        contract: String,
        amount: Uint128,
        msg: Binary,
    },
    /*Only with the "marketing" extension. If authorized, updates marketing metadata.
    Setting None/null for any of these will leave it unchanged.
    Setting Some("") will clear this field on the contract storage. */
    UpdateMarketing {
        /* A URL pointing to the project behind this token.*/
        project: Option<String>,
        /* A longer description of the token and it's utility. Designed for tooltips or such*/
        description: Option<String>,
        /* The address (if any) who can update this data structure*/
        marketing: Option<String>,
    },

    /* If set as the "marketing" role on the contract, upload a new URL, SVG, or PNG for the token*/
    UploadLogo(Logo),
}

#[derive(Serialize, Deserialize, Clone, Debug, PartialEq, JsonSchema)]
#[serde(rename_all = "snake_case")]
pub enum QueryMsg {
    /* Request stored token information */
    TokenConfig { },

    /* Default methods from CW20 Standard with no modifications:
    https://github.com/CosmWasm/cw-plus/tree/main/contracts/cw20-base */

    /* Returns the current balance of the given address, 0 if unset.
    Return type: BalanceResponse.*/
    Balance { address: String },

    /*Returns metadata on the contract - name, decimals, supply, etc.
    Return type: TokenInfoResponse.*/
    TokenInfo {},

    /*Only with "mintable" extension.
    Returns who can mint and the hard cap on maximum tokens after minting.
    Return type: MinterResponse.*/
    Minter {},

    /*Only with "allowance" extension.
    Returns how much spender can use from owner account, 0 if unset.
    Return type: AllowanceResponse.*/
    Allowance { owner: String, spender: String },

    /*Only with "enumerable" extension (and "allowances")
    Returns all allowances this owner has approved. Supports pagination.
    Return type: AllAllowancesResponse.*/
    AllAllowances {
        owner: String,
        start_after: Option<String>,
        limit: Option<u32>,
    },

    /*Only with "enumerable" extension
    Returns all accounts that have balances. Supports pagination.
    Return type: AllAccountsResponse.*/
    AllAccounts {
        start_after: Option<String>,
        limit: Option<u32>,
    },

    /*Only with "marketing" extension
    Returns more metadata on the contract to display in the client:
    - description, logo, project url, etc.
    Return type: MarketingInfoResponse*/
    MarketingInfo {},
    
    /*Only with "marketing" extension
    Downloads the mbeded logo data (if stored on chain). Errors if no logo data ftored for this
    contract.
    Return type: DownloadLogoResponse.*/
    DownloadLogo {},
}

#[derive(Serialize, Deserialize, Clone, Debug, PartialEq, JsonSchema)]
pub struct MigrateMsg {}