#[cfg(test)]
mod test {
    #[cfg(test)]
    use crate::contract::instantiate;
    use crate::{contract::execute, msg::{ExecuteMsg, InstantiateMsg}, ContractError};
    use cosmwasm_std::{
        testing::{mock_dependencies, mock_env, mock_info},
        DepsMut, Uint128,
    };
    use cw20::{Cw20Coin, MinterResponse, TokenInfoResponse};
    use cw20_base::{
        contract::{query_balance, query_token_info},
    };

    /* Generic method to instantiate the contract without repeating
    this code many times */
    fn do_instantiate(mut deps: DepsMut) -> TokenInfoResponse {
        // GIVEN
        let instantiate_msg = InstantiateMsg {
            token_factory_addr: "creator".to_string(),
            cw20 : cw20_base::msg::InstantiateMsg {
                name: "Bit Money".to_string(),
                symbol: "BTM".to_string(),
                decimals: 2,
                mint: Some(MinterResponse {
                    minter: "creator".to_string(),
                    cap: Some(Uint128::new(1234)),
                }),
                initial_balances: vec![Cw20Coin {
                    amount: Uint128::new(123),
                    address: "terra1x46rqay4d3cssq8gxxvqz8xt6nwlz4td20k38v".to_string(),
                }],
                marketing: None,
            }
        };
        let info = mock_info("creator", &[]);
        let env = mock_env();

        // WHEN
        let res = instantiate(deps.branch(), env, info, instantiate_msg).unwrap();

        // THEN
        assert_eq!(0, res.messages.len());
        let token_info = query_token_info(deps.as_ref()).unwrap();
        assert_eq!(
            token_info,
            TokenInfoResponse {
                name: "Bit Money".to_string(),
                symbol: "BTM".to_string(),
                decimals: 2,
                total_supply: Uint128::new(123),
            }
        );
        token_info
    }

    #[test]
    fn test_mint_new_token_units() {
        // GIVEN
        let mut deps = mock_dependencies(&[]);
        let info = mock_info("creator", &[]);
        let env = mock_env();
        let msg = ExecuteMsg::Mint {
            recipient: "creator".into(),
            amount: Uint128::new(123),
        };

        // WHEN
        do_instantiate(deps.as_mut());
        let res = execute(deps.as_mut(), env, info, msg).unwrap();

        // THEN
        assert_eq!(0, res.messages.len());
        assert_eq!(
            query_token_info(deps.as_ref()).unwrap(),
            TokenInfoResponse {
                name: "Bit Money".to_string(),
                symbol: "BTM".to_string(),
                decimals: 2,
                total_supply: Uint128::new(246),
            }
        );
        assert_eq!(
            query_balance(deps.as_ref(), "creator".to_string())
                .unwrap()
                .balance,
            Uint128::new(123)
        );
    }

    #[test]
    fn test_unauthorized_minting() {
        // GIVEN
        let mut deps = mock_dependencies(&[]);
        let info = mock_info("terra1x46rqay4d3cssq8gxxvqz8xt6nwlz4td20k38v", &[]);
        let env = mock_env();
        let mint_msg = ExecuteMsg::Mint {
            recipient: "terra1x46rqay4d3cssq8gxxvqz8xt6nwlz4td20k38v".into(),
            amount: Uint128::new(123),
        };

        // WHEN
        do_instantiate(deps.as_mut());
        let err = execute(deps.as_mut(), env, info, mint_msg).unwrap_err();

        // THEN
        assert_eq!(ContractError::Unauthorized {}, err);
    }
}
