use schemars::JsonSchema;
use serde::{Deserialize, Serialize};

use cosmwasm_std::{Binary, Uint128};
use cw20::{Expiration, Logo};

#[derive(Serialize, Deserialize, Clone, Debug, PartialEq, JsonSchema)]
#[serde(rename_all = "snake_case")]
pub struct InstantiateMsg {
    pub cw20: cw20_base::msg::InstantiateMsg,
    pub token_factory_addr: String
}

#[derive(Serialize, Deserialize, Clone, Debug, PartialEq, JsonSchema)]
#[serde(rename_all = "snake_case")]
pub enum ExecuteMsg {
    Mint { recipient: String, amount: Uint128 },

    // Implements CW20. Transfer is a base message to move tokens to another account without triggering actions
    Transfer { recipient: String, amount: Uint128 },

    // Burn is a message to destroy tokens forever
    Burn { amount: Uint128 },
    
    /* Implements CW20. Send is a base message to transfer tokens to a contract and trigger an action
    on the receiving contract.*/
    Send {
        contract: String,
        amount: Uint128,
        msg: Binary,
    },
    /* Implements CW20 "approval" extension. Allows spender to access an additional amount tokens
    from the owner's (env.sender) account. If expires is Some(), overwrites current allowance
    expiration with this one. */
    IncreaseAllowance {
        spender: String,
        amount: Uint128,
        expires: Option<Expiration>,
    },
    /* Implements CW20 "approval" extension. Lowers the spender's access of tokens
    from the owner's (env.sender) account by amount. If expires is Some(), overwrites current
    allowance expiration with this one. */
    DecreaseAllowance {
        spender: String,
        amount: Uint128,
        expires: Option<Expiration>,
    },
    /* Implements CW20 "approval" extension. Transfers amount tokens from owner -> recipient
    if `env.sender` has sufficient pre-approval. */
    TransferFrom {
        owner: String,
        recipient: String,
        amount: Uint128,
    },
    /* Implements CW20 "approval" extension. Sends amount tokens from owner -> contract
    if `env.sender` has sufficient pre-approval. */
    SendFrom {
        owner: String,
        contract: String,
        amount: Uint128,
        msg: Binary,
    },
    /*Only with the "marketing" extension. If authorized, updates marketing metadata.
    Setting None/null for any of these will leave it unchanged.
    Setting Some("") will clear this field on the contract storage. */
    UpdateMarketing {
        // A URL pointing to the project behind this token.
        project: Option<String>,
        // A longer description of the token and it's utility. Designed for tooltips or such
        description: Option<String>,
        // The address (if any) who can update this data structure
        marketing: Option<String>,
    },

    /// If set as the "marketing" role on the contract, upload a new URL, SVG, or PNG for the token
    UploadLogo(Logo),
}

#[derive(Serialize, Deserialize, Clone, Debug, PartialEq, JsonSchema)]
#[serde(rename_all = "snake_case")]
pub enum QueryMsg {
    // Implements CW20. Returns the current balance of the given address, 0 if unset.
    Balance { address: String },

    // Implements CW20. Returns metadata on the contract - name, decimals, supply, etc.
    TokenInfo {},
    
    /* Implements CW20 "allowance" extension. Returns how much 
    spender can use from owner account, 0 if unset. */
    Allowance { owner: String, spender: String },
}

#[derive(Serialize, Deserialize, Clone, Debug, PartialEq, JsonSchema)]
pub struct MigrateMsg {}