#[cfg(not(feature = "library"))]
use cosmwasm_std::entry_point;
use cosmwasm_std::{to_binary, Binary, Deps, DepsMut, Env, MessageInfo, Response, StdResult};
use cw2::set_contract_version;

use crate::error::ContractError;
use crate::msg::{ValueResponse, ExecuteMsg, InstantiateMsg, QueryMsg};
use crate::state::{State, STATE};

// version info for migration info
const CONTRACT_NAME: &str = "crates.io:cw-luke-vending-machine";
const CONTRACT_VERSION: &str = env!("CARGO_PKG_VERSION");

#[cfg_attr(not(feature = "library"), entry_point)]
pub fn instantiate(
    deps: DepsMut,
    _env: Env,
    info: MessageInfo,
    msg: InstantiateMsg,
) -> Result<Response, ContractError> {
    let state = State {
        cash: msg.cash,
        coins: msg.coins,
        price: msg.price,
        owner: info.sender.clone(),
    };
    set_contract_version(deps.storage, CONTRACT_NAME, CONTRACT_VERSION)?;
    STATE.save(deps.storage, &state)?;

    Ok(Response::new()
        .add_attribute("method", "instantiate")
        .add_attribute("owner", info.sender)
        .add_attribute("count", msg.cash.to_string()))
}

#[cfg_attr(not(feature = "library"), entry_point)]
pub fn execute(
    deps: DepsMut,
    _env: Env,
    info: MessageInfo,
    msg: ExecuteMsg,
) -> Result<Response, ContractError> {
    match msg {
        ExecuteMsg::Buy { cash } => buy(deps, info, cash),
        ExecuteMsg::Empty {} => empty(deps, info),
        ExecuteMsg::Refill { coins } => refill(deps, info, coins),
        ExecuteMsg::Set { price } => set(deps, info, price),
    }
}

pub fn empty(deps: DepsMut, info: MessageInfo) -> Result<Response, ContractError> {
    STATE.update(deps.storage, |mut state| -> Result<_, ContractError> {
        if info.sender != state.owner {
            return Err(ContractError::Unauthorized {});
        }
        state.cash = 0;
        Ok(state)
    })?;

    Ok(Response::new().add_attribute("method", "empty"))
}

pub fn buy(deps: DepsMut, info: MessageInfo, cash: i32) -> Result<Response, ContractError> {
    STATE.update(deps.storage, |mut state| -> Result<_, ContractError> {
        if info.sender != state.owner {
            return Err(ContractError::Unauthorized {});
        }
        state.cash += cash;
        state.coins -= cash / state.price; // TODO: check if empty
        Ok(state)
    })?;
    Ok(Response::new().add_attribute("method", "buy"))
}

pub fn refill(deps: DepsMut, info: MessageInfo, coins: i32) -> Result<Response, ContractError> {
    STATE.update(deps.storage, |mut state| -> Result<_, ContractError> {
        if info.sender != state.owner {
            return Err(ContractError::Unauthorized {});
        }
        state.coins += coins;
        Ok(state)
    })?;
    Ok(Response::new().add_attribute("method", "refill"))
}

pub fn set(deps: DepsMut, info: MessageInfo, price: i32) -> Result<Response, ContractError> {
    STATE.update(deps.storage, |mut state| -> Result<_, ContractError> {
        if info.sender != state.owner {
            return Err(ContractError::Unauthorized {});
        }
        state.price = price;
        Ok(state)
    })?;
    Ok(Response::new().add_attribute("method", "set"))
}

#[cfg_attr(not(feature = "library"), entry_point)]
pub fn query(deps: Deps, _env: Env, msg: QueryMsg) -> StdResult<Binary> {
    match msg {
        QueryMsg::GetCash {} => to_binary(&query_cash(deps)?),
        QueryMsg::GetCoins {} => to_binary(&query_coins(deps)?),
        QueryMsg::GetPrice {} => to_binary(&query_price(deps)?),
    }
}

fn query_cash(deps: Deps) -> StdResult<ValueResponse> {
    let state = STATE.load(deps.storage)?;
    Ok(ValueResponse { value: state.cash })
}

fn query_coins(deps: Deps) -> StdResult<ValueResponse> {
    let state = STATE.load(deps.storage)?;
    Ok(ValueResponse { value: state.coins })
}

fn query_price(deps: Deps) -> StdResult<ValueResponse> {
    let state = STATE.load(deps.storage)?;
    Ok(ValueResponse { value: state.price })
}

#[cfg(test)]
mod tests {
    use super::*;
    use cosmwasm_std::testing::{mock_dependencies, mock_env, mock_info};
    use cosmwasm_std::{coins, from_binary};

    #[test]
    fn proper_initialization() {
        // arrange
        let mut deps = mock_dependencies(&[]);
        let msg = InstantiateMsg { cash: 17, coins: 25, price: 10 };
        let info = mock_info("creator", &coins(1000, "earth"));

        // act
        let res = instantiate(deps.as_mut(), mock_env(), info, msg).unwrap();
        assert_eq!(0, res.messages.len());

        // assert
        let cash_response = query(deps.as_ref(), mock_env(), QueryMsg::GetCash {}).unwrap();
        let cash_value: ValueResponse = from_binary(&cash_response).unwrap();
        assert_eq!(17, cash_value.value);

        let coins_response = query(deps.as_ref(), mock_env(), QueryMsg::GetCoins {}).unwrap();
        let coins_value: ValueResponse = from_binary(&coins_response).unwrap();
        assert_eq!(25, coins_value.value);

        let price_response = query(deps.as_ref(), mock_env(), QueryMsg::GetPrice {}).unwrap();
        let price_value: ValueResponse = from_binary(&price_response).unwrap();
        assert_eq!(10, price_value.value);
    }

    #[test]
    fn empty() {
        // arrange
        let mut deps = mock_dependencies(&coins(2, "token"));
        let msg = InstantiateMsg { cash: 17, coins: 25, price: 10 };
        let info = mock_info("creator", &coins(2, "token"));
        let _res = instantiate(deps.as_mut(), mock_env(), info, msg).unwrap();

        // act - unauthorized
        let unauth_info = mock_info("anyone", &coins(2, "token"));
        let msg = ExecuteMsg::Empty {};
        let res = execute(deps.as_mut(), mock_env(), unauth_info, msg);

        // assert - unauthorized
        match res {
            Err(ContractError::Unauthorized {}) => {}
            _ => panic!("Must return unauthorized error"),
        }

        // act - authorized
        let auth_info = mock_info("creator", &coins(2, "token"));
        let msg = ExecuteMsg::Empty { };
        let _res = execute(deps.as_mut(), mock_env(), auth_info, msg).unwrap();

        // assert - authorized
        let res = query(deps.as_ref(), mock_env(), QueryMsg::GetCash {}).unwrap();
        let value: ValueResponse = from_binary(&res).unwrap();
        assert_eq!(0, value.value);
    }

    #[test]
    fn refill() {
        // arrange
        let mut deps = mock_dependencies(&coins(2, "token"));
        let msg = InstantiateMsg { cash: 17, coins: 25, price: 10 };
        let info = mock_info("creator", &coins(2, "token"));
        let _res = instantiate(deps.as_mut(), mock_env(), info, msg).unwrap();

        // act - unauthorized
        let unauth_info = mock_info("anyone", &coins(2, "token"));
        let msg = ExecuteMsg::Refill { coins: 15 };
        let res = execute(deps.as_mut(), mock_env(), unauth_info, msg);

        // assert - unauthorized
        match res {
            Err(ContractError::Unauthorized {}) => {}
            _ => panic!("Must return unauthorized error"),
        }

        // act - authorized
        let auth_info = mock_info("creator", &coins(2, "token"));
        let msg = ExecuteMsg::Refill { coins: 15 };
        let _res = execute(deps.as_mut(), mock_env(), auth_info, msg).unwrap();

        // assert - authorized
        let res = query(deps.as_ref(), mock_env(), QueryMsg::GetCoins {}).unwrap();
        let value: ValueResponse = from_binary(&res).unwrap();
        assert_eq!(40, value.value);
    }

    #[test]
    fn buy() {
        // arrange
        let mut deps = mock_dependencies(&coins(2, "token"));
        let msg = InstantiateMsg { cash: 17, coins: 25, price: 10 };
        let info = mock_info("creator", &coins(2, "token"));
        let _res = instantiate(deps.as_mut(), mock_env(), info, msg).unwrap();

        // act - unauthorized
        let unauth_info = mock_info("anyone", &coins(2, "token"));
        let msg = ExecuteMsg::Buy { cash: 31 };
        let res = execute(deps.as_mut(), mock_env(), unauth_info, msg);

        // assert - unauthorized
        match res {
            Err(ContractError::Unauthorized {}) => {}
            _ => panic!("Must return unauthorized error"),
        }

        // act - authorized
        let auth_info = mock_info("creator", &coins(2, "token"));
        let msg = ExecuteMsg::Buy { cash: 31 };
        let _res = execute(deps.as_mut(), mock_env(), auth_info, msg).unwrap();

        // assert - authorized
        let res = query(deps.as_ref(), mock_env(), QueryMsg::GetCoins {}).unwrap();
        let value: ValueResponse = from_binary(&res).unwrap();
        assert_eq!(22, value.value);
    }

    #[test]
    fn set() {
        // arrange
        let mut deps = mock_dependencies(&coins(2, "token"));
        let msg = InstantiateMsg { cash: 17, coins: 25, price: 10 };
        let info = mock_info("creator", &coins(2, "token"));
        let _res = instantiate(deps.as_mut(), mock_env(), info, msg).unwrap();

        // act - unauthorized
        let unauth_info = mock_info("anyone", &coins(2, "token"));
        let msg = ExecuteMsg::Set { price: 5 };
        let res = execute(deps.as_mut(), mock_env(), unauth_info, msg);

        // assert - unauthorized
        match res {
            Err(ContractError::Unauthorized {}) => {}
            _ => panic!("Must return unauthorized error"),
        }

        // act - authorized
        let auth_info = mock_info("creator", &coins(2, "token"));
        let msg = ExecuteMsg::Set { price: 5 };
        let _res = execute(deps.as_mut(), mock_env(), auth_info, msg).unwrap();

        // assert - authorized
        let res = query(deps.as_ref(), mock_env(), QueryMsg::GetPrice {}).unwrap();
        let value: ValueResponse = from_binary(&res).unwrap();
        assert_eq!(5, value.value);
    }
}
