use cursive_core::{
    Cursive,
    With,
    view::Nameable,
    event::{Event, Key},
    views::{EditView, Dialog, OnEventView, NamedView, Checkbox, LinearLayout, TextView},
    utils::markup::StyledString,
    theme::{
        PaletteColor,
        Color,
        BaseColor,
        ColorType,
        ColorStyle,
        BorderStyle,
        Theme
    }
};
use std::fmt::Display;

/// Convenience function that generates a better looking Cursive theme
/// 
/// # Example
/// ```
/// # use cursive_extras::*;
/// let mut root = cursive::default();
/// root.set_theme(better_theme());
/// ```
pub fn better_theme() -> Theme {
    let mut theme_def = Theme::default();
    theme_def.shadow = false;
    theme_def.palette[PaletteColor::Background] = Color::TerminalDefault;
    theme_def.palette[PaletteColor::Primary] = Color::Light(BaseColor::White);
    theme_def.palette[PaletteColor::View] = Color::TerminalDefault;
    theme_def.palette[PaletteColor::Highlight] = Color::Light(BaseColor::Blue);
    theme_def.palette[PaletteColor::HighlightText] = Color::Dark(BaseColor::White);
    theme_def.palette[PaletteColor::Secondary] = Color::Dark(BaseColor::Blue);
    theme_def.palette[PaletteColor::TitlePrimary] = Color::Light(BaseColor::Blue);
    theme_def.borders = BorderStyle::Outset;
    theme_def
}

/// Convenience function that creates a dialog that runs a callback if the
/// user selects "Yes"
/// 
/// # Example
/// ```
/// # use cursive_extras::*;
/// let mut root = cursive::default();
/// # root.set_theme(better_theme());
/// root.add_layer(confirm_dialog("Are you sure?", "Are you sure?", |view| view.quit()));
/// root.run();
/// ```
pub fn confirm_dialog<C, T, U>(title: T, text: U, cb: C) -> OnEventView<Dialog>
    where
        C: Fn(&mut Cursive) + 'static,
        T: Display,
        U: Display
{
    Dialog::text(text.to_string())
        .dismiss_button("No")
        .button("Yes", cb)
        .title(title.to_string())
        .wrap_with(OnEventView::new)
        .on_event(Event::Key(Key::Esc), |v| { v.pop_layer(); })
}

/// Same as `confirm_dialog()`, but accepts `StyledString`s instead for the text
pub fn confirm_dialog_styled<T:Display, C: Fn(&mut Cursive) + 'static>(title: T, text: StyledString, cb: C) -> OnEventView<Dialog> {
    Dialog::text(text)
        .dismiss_button("No")
        .button("Yes", cb)
        .title(title.to_string())
        .wrap_with(OnEventView::new)
        .on_event(Event::Key(Key::Esc), |v| { v.pop_layer(); })
}

/// Convenience function that shows a user a dialog box
/// with a message that includes a back button
/// 
/// # Example
/// ```
/// # use cursive_extras::*;
/// let mut root = cursive::default();
/// # root.set_theme(better_theme());
/// root.add_layer(info_dialog("Info", "This is important!"));
/// root.run();
/// ```
pub fn info_dialog<T: Display, U: Display>(title: T, text: U) -> OnEventView<Dialog> {
    Dialog::text(text.to_string())
        .dismiss_button("Back")
        .title(title.to_string())
        .wrap_with(OnEventView::new)
        .on_event(Event::Key(Key::Esc), |v| { v.pop_layer(); })
}

/// Sames as `info_dialog()`, but accepts `StyledString`s instead for the text
pub fn info_dialog_styled<T: Display>(title: T, text: StyledString) -> OnEventView<Dialog> {
    Dialog::text(text)
        .dismiss_button("Back")
        .title(title.to_string())
        .wrap_with(OnEventView::new)
        .on_event(Event::Key(Key::Esc), |v| { v.pop_layer(); })
}

/// Convenience function that creates a named `EditView` that has a better
/// looking style and can optionally act as a password entry box
/// 
/// # Example
/// ```
/// # use cursive::views::Dialog;
/// # use cursive_extras::*;
/// let mut root = cursive::default();
/// # root.set_theme(better_theme());
/// root.add_fullscreen_layer(
///     Dialog::around(styled_editview("yes", "edit", false))
///         .button("Quit", |view| view.quit())
///         .title("Styled EditView Example")
/// );
/// root.run();
/// ```
pub fn styled_editview<C: Display>(content: C, view_name: &'static str, password: bool) -> NamedView<EditView> {
    styled_editview_color(content, view_name, password, PaletteColor::Highlight)
}

/// Same as `styled_editview()` but allows for a color to be chosen instead of using the highlight color
pub fn styled_editview_color<T: Display, C: Into<ColorType>>(content: T, view_name: &'static str, password: bool, color: C) -> NamedView<EditView> {
    let input_style = ColorStyle::new(
        color,
        Color::Light(BaseColor::White),
    );
    let view = EditView::new().content(content.to_string()).style(input_style).filler(" ");

    if password {
        view.secret().with_name(view_name)
    }
    else {
        view.with_name(view_name)
    }
}

/// Convenience function that return the state of a named check box
/// 
/// true: the box is checked
/// 
/// false: of course it isn't checked!
pub fn get_checkbox_option(view: &mut Cursive, name: &str) -> bool {
    let cbox = view.find_name::<Checkbox>(name).unwrap();
    cbox.is_checked()
}

/// Convenience function that returns a horizontal `LinearLayout` that is a named check box with a label
pub fn labeled_checkbox(text: &str, name: &str, checked: bool) -> LinearLayout {
    labeled_checkbox_cb(text, name, checked, |_, _| { })
}

/// Same as `labeled_checkbox()` but also accepts a closure to execute when the check box's state changes
pub fn labeled_checkbox_cb<C: Fn(&mut Cursive, bool) + 'static>(text: &str, name: &str, checked: bool, callback: C) -> LinearLayout {
    LinearLayout::horizontal()
        .child(Checkbox::new().with_checked(checked).on_change(callback).with_name(name))
        .child(TextView::new(format!(" {}", text)))
}