use std::{
    cell::RefCell,
    rc::Rc,
    thread::{self, JoinHandle}
};
use crossbeam_channel::{bounded, Receiver};
use cursive_core::{
    Printer, Vec2,
    event::{Event, EventResult},
    view::View,
    theme::ColorStyle
};

type WorkerThread<T> = Rc<RefCell<Option<JoinHandle<T>>>>;

/// Simple loading animation view that runs a task in a background thread and shows a loading animation and message while it completes and allows for retrieval of a return value
/// 
/// # Example
/// ```
/// # use cursive_extras::*;
/// # use cursive::{
/// #   views::{Dialog, LinearLayout},
/// #   traits::Nameable
/// # };
/// let mut root = cursive::default();
/// # root.set_theme(better_theme());
/// let anim = LoadingAnimation::new("Loading Dummy Resource...", || {
///     thread::sleep(Duration::from_secs(10));
/// });
/// 
/// root.add_layer(Dialog::around(anim.with_name("loader")).title("Loading..."));
/// root.set_fps(30);

/// root.set_global_callback(Event::Refresh, |view| {
///     let mut loader = view.find_name::<LoadingAnimation<()>>("loader").unwrap();
///     if loader.is_done() {
///         loader.finish().unwrap_or(());
///         view.quit()
///     }
/// });
/// root.run();
/// ```
#[derive(Clone)]
pub struct LoadingAnimation<T: Send + Sync + 'static> {
    worker: WorkerThread<T>,
    recv: Receiver<bool>,
    completed: bool,
    message: String,
    width: usize,
    anim_x: usize,
    reversed: bool
}

impl<T: Send + Sync> LoadingAnimation<T> {
    /// Create a new `LoadingAnimation` with the specified closure or function pointer as the task
    pub fn new<U>(message: &str, task: U) -> LoadingAnimation<T> 
        where U: FnOnce() -> T + Send + Sync + 'static
    {
        let (sender, recv) = bounded::<bool>(0);
        let worker = Rc::new(RefCell::from(
            Some(
                thread::spawn(move || {
                    let out = task();
                    sender.send(true).expect("Did the other side disconnect?");
                    out
                })
            )
        ));
        LoadingAnimation {
            worker,
            recv,
            completed: false,
            width: message.chars().count(),
            message: message.to_string(),
            anim_x: 0,
            reversed: false
        }
    }

    /// Has the background task finished executing?
    pub fn is_done(&mut self) -> bool {
        if let Ok(b) = self.recv.try_recv() {
            self.completed = b;
            return b;
        }
        else if self.completed {
            return true;
        }
        if let None = *self.worker.borrow() {
            self.completed = true;
            true
        }
        else {
            false
        }
    }

    /// Join with the task's thread, block until it is finished, and return the resulting value
    /// 
    /// It is best to run this when `LoadingAnimation::is_done()` is true
    /// 
    /// This will return `None` if called more than once
    /// 
    /// # Panics
    /// This method will panic if the underlying task panicked while executing
    pub fn finish(&mut self) -> Option<T> {
        if self.worker.borrow().is_none() {
            None
        }
        else {
            let worker = self.worker.borrow_mut().take().unwrap();
            Some(worker.join().unwrap())
        }
    }
}

impl<T: Send + Sync> View for LoadingAnimation<T> {
    fn draw(&self, printer: &Printer) {
        let style = ColorStyle::secondary();
        printer.print((0, 0), &" ".repeat(self.width));
        printer.with_color(style, |printer| {
            printer.print((0, 0), &format!("{}███", " ".repeat(self.anim_x)));
        });
        printer.print((0, 1), &self.message);
    }

    fn on_event(&mut self, event: Event) -> EventResult {
        if let Event::Refresh = event {
            if self.width == 0 {
                return EventResult::Ignored;
            }
            else if self.reversed {
                if self.anim_x == 0 {
                    self.reversed = false;
                }
                else {
                    self.anim_x -= 1;
                }
            }
            else {
                if self.anim_x < self.width - 3 {
                    self.anim_x += 1;
                }
                else {
                    self.reversed = true;
                }
            }
        }
        EventResult::Ignored
    }

    fn required_size(&mut self, _: Vec2) -> Vec2 {
        Vec2::new(self.message.chars().count(), 2)
    }
}