#![allow(unused_imports)]
use crate::*;
use std::{
    thread,
    time::Duration,
    sync::Mutex
};
use cursive::{
    views::{
        Dialog,
        LinearLayout,
        TextView,
        SelectView,
        EditView
    },
    event::Event,
    traits::Nameable
};
use lazy_static::lazy_static;

lazy_static! {
    // this make sure each test runs without overlapping others
    static ref RUN: Mutex<()> = Mutex::from(());
}

// horizontal divider test
#[test]
fn hdivider() {
    let runlock = RUN.lock().unwrap();
    let mut root = cursive::default();
    root.set_theme(better_theme());

    root.add_fullscreen_layer(
        Dialog::around(
            LinearLayout::horizontal()
                .child(HDivider::new())
                .child(TextView::new("Is there a line to the left of this?"))
        )
        .button("Yes", |v| v.quit())
        .button("No", |_| panic!("HDivider test failed!"))
        .title("Coof Testing")
    );
    root.run();
    drop(runlock);
}

// vertical divider test
#[test]
fn vdivider() {
    let runlock = RUN.lock().unwrap();
    let mut root = cursive::default();
    root.set_theme(better_theme());
    root.add_fullscreen_layer(
        Dialog::around(
            LinearLayout::vertical()
                .child(VDivider::new())
                .child(TextView::new("Is there a line above this?"))
        )
        .button("Yes", |v| v.quit())
        .button("No", |_| panic!("VDivider test failed!"))
        .title("Coof Testing")
    );
    root.run();
    drop(runlock);
}

// status bar test 1
#[test]
fn statbar1() {
    let runlock = RUN.lock().unwrap();
    let mut root = cursive::default();
    root.set_theme(better_theme());
    root.add_fullscreen_layer(
        LinearLayout::vertical()
        .child(
            Dialog::text("Is there a message below this?")
                .button("Yes", |v| v.quit())
                .button("No", |_| panic!("StatusView test 1 failed!"))
                .title("Coof Testing")
        )
        .child(StatusView::new().with_name("status"))
    );
    root.set_fps(30);
    root.set_global_callback(Event::Refresh, |view| {
        let mut status = view.find_name::<StatusView>("status").expect("StatusView test 1 failed!");
        status.info("yes");
        status.update();
    });
    root.run();
    drop(runlock);
}

// status bar test 2
#[test]
fn statbar2() {
    let runlock = RUN.lock().unwrap();
    let mut root = cursive::default();
    root.set_theme(better_theme());
    root.add_fullscreen_layer(
        LinearLayout::vertical()
        .child(
            Dialog::text("Is there an error message below this?")
                .button("Yes", |v| v.quit())
                .button("No", |_| panic!("StatusView test 1 failed!"))
                .title("Coof Testing")
        )
        .child(StatusView::new().with_name("status"))
    );
    root.set_fps(30);
    root.set_global_callback(Event::Refresh, |view| {
        let mut status = view.find_name::<StatusView>("status").expect("StatusView test 2 failed!");
        status.report_error("Error: no");
        status.update();
    });
    root.run();
    drop(runlock);
}

// status bar test 3
#[test]
fn statbar3() {
    let runlock = RUN.lock().unwrap();
    let mut root = cursive::default();
    root.set_theme(better_theme());
    root.add_fullscreen_layer(
        LinearLayout::vertical()
        .child(
            Dialog::text("Is there another error message below this?")
                .button("Yes", |v| v.quit())
                .button("No", |_| panic!("StatusView test 3 failed!"))
                .title("Coof Testing")
        )
        .child(StatusView::new().with_name("status"))
    );
    root.set_fps(30);
    root.set_global_callback(Event::Refresh, |view| {
        let error: Result<&str, &str> = Err("Error: Houston, we have a problem!");
        let mut status = view.find_name::<StatusView>("status").expect("StatusView test 3 failed!");
        report_error!(status, error);
        status.update();
    });
    root.run();
    drop(runlock);
}

// select_view! macro test
#[test]
fn svmacro() {
    let runlock = RUN.lock().unwrap();
    let mut root = cursive::default();
    root.set_theme(better_theme());
    let sv = select_view! {
        "yes" => true,
        "no" => false
    }.on_submit(|v, _| v.quit());
    root.add_fullscreen_layer(Dialog::around(sv).title("Coof Testing"));
    root.run();
    drop(runlock);
}

// info dialog test and styled edit view test
#[test]
fn info_dialogt() {
    let runlock = RUN.lock().unwrap();
    let mut root = cursive::default();
    root.set_theme(better_theme());
    root.add_fullscreen_layer(
        Dialog::around(styled_editview("coof", "edit", false))
            .button("Info", |view| {
                let edit_view = view.find_name::<EditView>("edit").unwrap();
                view.add_layer(info_dialog("Info", edit_view.get_content()));
            })
            .button("Quit", |view| {
                view.add_layer(confirm_dialog("Are you sure?", "Are you sure?", |view| view.quit()));
            })
            .title("Coof Testing")
    );
    root.run();
    drop(runlock);
}

// settings! macro test
#[test]
fn settings1() {
    let runlock = RUN.lock().unwrap();
    let mut root = cursive::default();
    root.set_theme(better_theme());
    root.add_fullscreen_layer(
        settings_cb!(
            "Coof Testing",
            "Test",
            |view| view.quit(),
            TextView::new("Variant:"),
            EditView::new().content("delta")
        )
    );
    root.run();
    drop(runlock);
}

#[test]
fn settings2() {
    let runlock = RUN.lock().unwrap();
    let mut root = cursive::default();
    root.set_theme(better_theme());
    root.add_fullscreen_layer(
        settings!(
            "Coof Testing",
            |view| view.quit(),
            TextView::new("Variant:"),
            EditView::new().content("delta")
        )
    );
    root.run();
    drop(runlock);
}

// hlayout! macro test
#[test]
fn hlayout() {
    let runlock = RUN.lock().unwrap();
    let mut root = cursive::default();
    root.set_theme(better_theme());
    root.add_fullscreen_layer(
        Dialog::around(
            hlayout!(
                TextView::new("Option:"),
                EditView::new()
            )
        )
        .button("Quit", |view| view.quit())
    );
    root.run();
    drop(runlock);
}

// vlayout! macro test
#[test]
fn vlayout() {
    let runlock = RUN.lock().unwrap();
    let mut root = cursive::default();
    root.set_theme(better_theme());
    root.add_fullscreen_layer(
        Dialog::around(
            vlayout!(
                TextView::new("Option:"),
                EditView::new()
            )
        )
        .button("Quit", |view| view.quit())
    );
    root.run();
    drop(runlock);
}

// loading animation test 1
#[test]
fn loadanimation1() {
    let runlock = RUN.lock().unwrap();
    let mut root = cursive::default();
    root.set_theme(better_theme());
    let anim = LoadingAnimation::new("This is a long computation...", || {
        thread::sleep(Duration::from_secs(10));
    });
    root.add_layer(Dialog::around(anim.with_name("loader")).title("Loading Animation Test").with_name("dialog"));
    root.set_fps(30);

    root.set_global_callback(Event::Refresh, |view| {
        let mut loader = view.find_name::<LoadingAnimation<()>>("loader").expect("StatusView test 3 failed!");
        let mut dialog = view.find_name::<Dialog>("dialog").expect("StatusView test 3 failed!");
        if loader.is_done() {
            loader.finish().unwrap_or(());
            if dialog.buttons_len() == 0 {
                dialog.set_title("Loading Animation Test (Completed)");
                dialog.add_button("Quit", |view| view.quit());
            }
        }
    });
    root.run();
    drop(runlock);
}

// loading animation test 2 (simulated loading pop up)
#[test]
fn loadanimation2() {
    let runlock = RUN.lock().unwrap();
    let mut root = cursive::default();
    root.set_theme(better_theme());
    let anim = LoadingAnimation::new("Loading Dummy Resource...", || {
        thread::sleep(Duration::from_secs(10));
    });
    root.add_layer(Dialog::around(anim.with_name("loader")).title("Loading...").with_name("dialog"));
    root.set_fps(30);

    root.set_global_callback(Event::Refresh, |view| {
        let mut loader = view.find_name::<LoadingAnimation<()>>("loader").expect("StatusView test 3 failed!");
        if loader.is_done() {
            loader.finish().unwrap_or(());
            view.quit()
        }
    });
    root.run();
    drop(runlock);
}

// loading animation test 3 (simulated loading pop up and return value)
#[test]
fn loadanimation3() {
    let runlock = RUN.lock().unwrap();
    let mut root = cursive::default();
    root.set_theme(better_theme());
    let anim = LoadingAnimation::new("Loading Dummy Number...", || {
        thread::sleep(Duration::from_secs(10));
        2 + 4
    });
    root.add_layer(Dialog::around(anim.with_name("loader")).title("Loading...").with_name("dialog"));
    root.set_fps(30);

    root.set_global_callback(Event::Refresh, |view| {
        let mut loader = view.find_name::<LoadingAnimation<i32>>("loader").expect("StatusView test 3 failed!");
        if loader.is_done() {
            let ret_val = loader.finish().unwrap();
            assert_eq!(ret_val, 6);
            view.quit()
        }
    });
    root.run();
    drop(runlock);
}