#![allow(unused_imports)]
#[cfg(test)]
use crate::*;

#[test]
fn tests() {
    use cursive::{
        views::{
            Dialog,
            LinearLayout,
            TextView,
            SelectView,
            EditView
        },
        event::Event,
        traits::Nameable
    };

    // horizontal divider test
    let mut root = cursive::default();
    root.set_theme(better_theme());
    root.add_fullscreen_layer(
        Dialog::around(
            LinearLayout::horizontal()
                .child(HDivider::new())
                .child(TextView::new("Is there a line to the left of this?"))
        )
        .button("Yes", |v| v.quit())
        .button("No", |_| panic!("HDivider test failed!"))
        .title("Coof Testing")
    );
    root.run();

    // vertical divider test
    root.add_fullscreen_layer(
        Dialog::around(
            LinearLayout::vertical()
                .child(VDivider::new())
                .child(TextView::new("Is there a line above this?"))
        )
        .button("Yes", |v| v.quit())
        .button("No", |_| panic!("VDivider test failed!"))
        .title("Coof Testing")
    );
    root.run();

    // status bar test 1
    root.add_fullscreen_layer(
        LinearLayout::vertical()
        .child(
            Dialog::text("Is there a message below this?")
                .button("Yes", |v| v.quit())
                .button("No", |_| panic!("StatusView test 1 failed!"))
                .title("Coof Testing")
        )
        .child(StatusView::new().with_name("status"))
    );
    root.set_fps(30);
    root.set_global_callback(Event::Refresh, |view| {
        let mut status = view.find_name::<StatusView>("status").expect("StatusView test 1 failed!");
        status.info("yes");
        status.update();
    });
    root.run();

    // status bar test 2
    root.add_fullscreen_layer(
        LinearLayout::vertical()
        .child(
            Dialog::text("Is there an error message below this?")
                .button("Yes", |v| v.quit())
                .button("No", |_| panic!("StatusView test 1 failed!"))
                .title("Coof Testing")
        )
        .child(StatusView::new().with_name("status"))
    );
    root.set_fps(30);
    root.set_global_callback(Event::Refresh, |view| {
        let mut status = view.find_name::<StatusView>("status").expect("StatusView test 2 failed!");
        status.report_error("Error: no");
        status.update();
    });
    root.run();

    // status bar test 3
    root.add_fullscreen_layer(
        LinearLayout::vertical()
        .child(
            Dialog::text("Is there another error message below this?")
                .button("Yes", |v| v.quit())
                .button("No", |_| panic!("StatusView test 3 failed!"))
                .title("Coof Testing")
        )
        .child(StatusView::new().with_name("status"))
    );
    root.set_fps(30);
    root.set_global_callback(Event::Refresh, |view| {
        let error: Result<&str, &str> = Err("Error: Houston, we have a problem!");
        let mut status = view.find_name::<StatusView>("status").expect("StatusView test 3 failed!");
        report_error!(status, error);
        status.update();
    });
    root.run();

    // SelectView macro test
    let sv = select_view! {
        "yes" => true,
        "no" => false
    }.on_submit(|v, _| v.quit());
    root.add_fullscreen_layer(Dialog::around(sv).title("Coof Testing"));
    root.run();

    // info dialog test and styled edit view test
    root.add_fullscreen_layer(
        Dialog::around(styled_editview("coof", "edit", false))
            .button("Info", |view| {
                let edit_view = view.find_name::<EditView>("edit").unwrap();
                view.add_layer(info_dialog("Info", edit_view.get_content()));
            })
            .button("Quit", |view| {
                view.add_layer(confirm_dialog("Are you sure?", "Are you sure?", |view| view.quit()));
            })
            .title("Coof Testing")
    );
    root.run();

    // settings macro test
    root.add_fullscreen_layer(
        settings!(
            "Coof Testing",
            |view| view.quit(),
            TextView::new("Option:"),
            EditView::new()
        )
    );
    root.run();

    // hlayout macro test
    root.add_fullscreen_layer(
        Dialog::around(
            hlayout!(
                TextView::new("Option:"),
                EditView::new()
            )
        )
        .button("Quit", |view| view.quit())
    );
    root.run();

    // vlayout macro test
    root.add_fullscreen_layer(
        Dialog::around(
            vlayout!(
                TextView::new("Option:"),
                EditView::new()
            )
        )
        .button("Quit", |view| view.quit())
    );
    root.run();
}