use cursive_core::{
    theme::{Color, ColorStyle, BaseColor},
    View,
    Printer,
    Vec2
};
use std::{
    time::Instant,
    fmt::Display
};

// to do: multiline button and table view

/// Horizontal divider view
/// 
/// # Example
/// ```
/// # use cursive_extras::*;
/// # use cursive::views::{Dialog, LinearLayout}
/// let mut root = cursive::default();
/// root.add_fullscreen_layer(
///     Dialog::around(
///         LinearLayout::horizontal()
///             .child(select_view!{
///                 "item1" => 1,
///                 "item2" => 2,
///                 "item3" => 3,
///              })
///             .child(HDivider::new())
///             .child(select_view!{
///                 "item4" => 4,
///                 "item5" => 5,
///                 "item6" => 6,
///              })
///     )
///     .button("Quit", |v| v.quit())
///     .title("Horizontal Divider Example")
/// );
/// root.run();
/// ```
#[derive(Copy, Clone)]
pub struct HDivider(usize);

impl HDivider {
    /// Creates new horizontal divider
    pub fn new() -> HDivider {
        HDivider(0)
    }
}

impl View for HDivider {
    fn draw(&self, printer: &Printer) {
        for i in 0..self.0 {
            printer.print((0, i), "│")
        }
    }

    fn required_size(&mut self,constraint: Vec2) -> Vec2 {
        self.0 = constraint.y;
        Vec2::new(1, constraint.y)
    }
}

/// Vertical divider view
/// 
/// # Example
/// ```
/// # use cursive_extras::*;
/// # use cursive::views::{Dialog, LinearLayout}
/// let mut root = cursive::default();
/// root.add_fullscreen_layer(
///     Dialog::around(
///         LinearLayout::vertical()
///             .child(select_view!{
///                 "item1" => 1,
///                 "item2" => 2,
///                 "item3" => 3,
///              })
///             .child(VDivider::new())
///             .child(select_view!{
///                 "item4" => 4,
///                 "item5" => 5,
///                 "item6" => 6,
///              })
///     )
///     .button("Quit", |v| v.quit())
///     .title("Vertical Divider Example")
/// );
/// root.run();
/// ```
#[derive(Copy, Clone)]
pub struct VDivider(usize);

impl VDivider {
    /// Creates new vertical divider
    pub fn new() -> VDivider {
        VDivider(0)
    }
}

impl View for VDivider {
    fn draw(&self, printer: &Printer) {
        for i in 0..self.0 {
            printer.print((i, 0), "─")
        }
    }

    fn required_size(&mut self,constraint: Vec2) -> Vec2 {
        self.0 = constraint.x;
        Vec2::new(constraint.x, 1)
    }
}

/// View that can be used to report an application's status. 
/// It is meant to be placed at the bottom of the main Cursive layer
/// 
/// # Examples
/// 
/// ## Reporting Application Status
/// ```
/// # use cursive_extras::*;
/// # use cursive::{
/// #   views::{Dialog, LinearLayout},
/// #   view::Nameable,
/// #   event::Event
/// # }
/// let mut root = cursive::default();
/// root.add_fullscreen_layer(
///     LinearLayout::vertical()
///         .child(
///             Dialog::text("Yes")
///                 .button("Quit", |v| v.quit())
///                 .title("StatusView Example")
///         )
///         .child(StatusView::new().with_name("status"))
/// );
/// root.set_fps(30);
/// root.set_global_callback(Event::Refresh, |view| {
///     let mut status = view.find_name::<StatusView>("status").expect("StatusView does not exist!");
///     status.info("Application Status");
///     status.update();
/// });
/// root.run();
/// ```
/// 
/// ## Reporting an Error
/// ```
/// # use cursive_extras::*;
/// # use cursive::{
/// #   views::{Dialog, LinearLayout},
/// #   view::Nameable,
/// #   event::Event
/// # } 
/// root.add_fullscreen_layer(
///     LinearLayout::vertical()
///         .child(
///             Dialog::text("Yes")
///                 .button("Quit", |v| v.quit())
///                 .title("StatusView Example")
///         )
///         .child(StatusView::new().with_name("status"))
/// );
/// 
/// root.set_fps(30);
/// root.set_global_callback(Event::Refresh, |view| {
///     let error: Result<&str, &str> = Err("Error: Houston, we have a problem!");
///     let mut status = view.find_name::<StatusView>("status").expect("StatusView does not exist!");
///     report_error!(status, error);
///     status.update();
/// });
/// root.run();
/// ```
pub struct StatusView {
    cur_msg: String,
    time: Instant,
    error: bool
}

impl StatusView {
    /// Create a new `StatusView`
    pub fn new() -> StatusView {
        StatusView {
            cur_msg: String::new(),
            time: Instant::now(),
            error: false
        }
    }

    /// This will clear the `StatusView` if called more than 5 seconds after the text
    /// was set
    pub fn update(&mut self) {
        if self.time.elapsed().as_secs() > 5 {
            self.cur_msg = String::new();
            self.time = Instant::now();
            self.error = false;
        }
    }

    /// Set the text in the `StatusView` with error formatting (bright red text)
    pub fn report_error<T: Display>(&mut self, text: T) {
        self.cur_msg = text.to_string();
        self.time = Instant::now();
        self.error = true;
    }

    /// Set the text in the `StatusView` with normal formatting (white text)
    pub fn info<T: Display>(&mut self, text: T) {
        self.cur_msg = text.to_string();
        self.time = Instant::now();
    }
}

impl View for StatusView {
    fn draw(&self, printer: &Printer) {
        if printer.size.x == 0 && printer.size.y == 0 && !self.cur_msg.is_empty() {
            return;
        }
        let text_color = ColorStyle::from(
            if self.error {
                Color::Light(BaseColor::Red)
            }
            else {
                Color::Light(BaseColor::White)
            }
        );

        printer.with_color(text_color, |printer| printer.print((0, 0), &self.cur_msg))
    }

    fn required_size(&mut self, constraint: Vec2) -> Vec2 {
        let y = !self.cur_msg.is_empty() as usize;
        Vec2::new(constraint.x, y)
    }
}