/// Convenience macro that updates the specified `StatusView` reference with an error message and returns 
/// from the current callback if an error happens, returns the `Ok` value otherwise
/// 
/// # Example
/// ```
/// # use cursive::views::{LinearLayout, Dialog};
/// # use cursive_extras::*;
/// # use cursive::event::Event;
/// # use cursive::view::Nameable;
/// let mut root = cursive::default();
/// root.add_fullscreen_layer(
///     LinearLayout::vertical()
///         .child(
///             Dialog::text("Yes")
///                 .button("Quit", |v| v.quit())
///                 .title("Error Reporting example")
///         )
///         .child(StatusView::new().with_name("status"))
/// );
/// root.set_fps(30);
/// root.set_global_callback(Event::Refresh, |view| {
///     let error: Result<&str, &str> = Err("Error: Houston, we have a problem!");
///     let mut status = view.find_name::<StatusView>("status").expect("StatusView does not exist!");
///     report_error!(status, error);
///     status.update();
/// });
/// root.run();
/// ```
#[macro_export]
macro_rules! report_error {
    ($status_view:expr, $res_val:expr) => {
        match $res_val {
            Ok(v) => v,
            Err(error) => {
                $status_view.report_error(error);
                return;
            }
        }
    }
}

/// Convenience macro that makes `SelectView`s easier to create
/// 
/// # Example
/// ```
/// # use cursive_extras::*;
/// # use cursive::views::{Dialog, SelectView};
/// let sv = select_view! {
///     "yes" => true,
///     "no" => false
/// }.on_submit(|v, _| v.quit());
/// 
/// let mut root = cursive::default();
/// root.add_fullscreen_layer(Dialog::around(sv).title("Select View"));
/// root.run();
/// ```
#[macro_export]
macro_rules! select_view {
    ($($label:expr => $item:expr), *) => {
        {
            use cursive_core::views::SelectView;
            let mut temp_sv = SelectView::new();
            $(
                temp_sv.add_item($label, $item);
            )*
            temp_sv
        }
    };
}

/// Convenience macro that generates a settings view using sub-views wrapped in a `Dialog` view
///
///  # Example
/// ```
/// # use cursive::views::{TextView, EditView};
/// # use cursive_extras::*;
/// let mut root = cursive::default();
/// root.add_fullscreen_layer(
///     settings!(
///         "Settings Example", // title
///         |view| view.quit(), // callback if "Save" is selected
///         TextView::new("Option:"), // views
///         EditView::new()
///     )
/// );
/// root.run();
/// ```
#[macro_export]
macro_rules! settings {
    ($title:expr, $save_cb:expr, $($view:expr), *) => {
        {
            use cursive_core::views::{Dialog, LinearLayout};
            Dialog::around(
                LinearLayout::vertical()
                $(
                    .child($view)
                )*
            )
            .dismiss_button("Back")
            .button("Save", $save_cb)
            .title($title)
        }
    };
}

/// Convenience macro that creates a horizontal layout of views
///
///  # Example
/// ```
/// # use cursive_extras::*;
/// # use cursive::views::{Dialog, TextView};
/// let mut root = cursive::default();
/// root.add_fullscreen_layer(
///     Dialog::around(
///         hlayout!(
///             TextView::new("Yes "),
///             TextView::new("No")
///         )
///     )
///     .button("Quit", |view| view.quit())
/// );
/// root.run();
/// ```
#[macro_export]
macro_rules! hlayout {
    ($($view:expr), *) => {
        {
            use cursive_core::views::LinearLayout;
            LinearLayout::horizontal()
            $(
                .child($view)
            )*
        }
    };
}

/// Convenience macro that creates a vertical layout of views
/// 
///  # Example
/// ```
/// # use cursive_extras::*;
/// # use cursive::views::{Dialog, TextView};
/// let mut root = cursive::default();
/// root.add_fullscreen_layer(
///     Dialog::around(
///         vlayout!(
///             TextView::new("Yes "),
///             TextView::new("No")
///         )
///     )
///     .button("Quit", |view| view.quit())
/// );
/// root.run();
/// ```
#[macro_export]
macro_rules! vlayout {
    ($($view:expr), *) => {
        {
            use cursive_core::views::LinearLayout;
            LinearLayout::vertical()
            $(
                .child($view)
            )*
        }
    };
}