// Copyright (C) 2021 Robin Krahl <robin.krahl@ireas.org>
// SPDX-License-Identifier: Apache-2.0 or MIT

use std::fmt;

use crate::{command::Command, error::Error, hid::Device, message};

#[derive(Clone, Copy, Debug, Eq, Ord, PartialEq, PartialOrd)]
pub struct Channel(u32);

impl Channel {
    pub const BROADCAST: Channel = Channel(0xffff_ffff);
}

impl From<u32> for Channel {
    fn from(id: u32) -> Self {
        Self(id)
    }
}

impl From<Channel> for u32 {
    fn from(channel: Channel) -> u32 {
        channel.0
    }
}

impl From<[u8; 4]> for Channel {
    fn from(data: [u8; 4]) -> Self {
        Self::from(u32::from_be_bytes(data))
    }
}

impl fmt::Display for Channel {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "0x{:x}", self.0)
    }
}

pub fn exec(
    device: &impl Device,
    channel: Channel,
    command: Command,
    data: &[u8],
) -> Result<Vec<u8>, Error> {
    let mut buffer = vec![0; device.packet_size() + 1];
    message::send(device, channel, command, data, &mut buffer)?;
    message::receive(device, channel, command, &mut buffer)
}
