// Copyright (C) 2021 Robin Krahl <robin.krahl@ireas.org>
// SPDX-License-Identifier: Apache-2.0 or MIT

use crate::error::{Error, RequestError};

pub trait Device {
    fn packet_size(&self) -> usize {
        64
    }

    fn send(&self, data: &[u8]) -> Result<(), Error>;

    fn receive<'a>(&self, buffer: &'a mut [u8]) -> Result<&'a [u8], Error>;
}

impl Device for hidapi::HidDevice {
    fn send(&self, data: &[u8]) -> Result<(), Error> {
        log::trace!("Sending {} bytes: {:x?}", data.len(), data);
        let n = self.write(data)?;
        if n == data.len() {
            Ok(())
        } else {
            Err(Error::from(RequestError::IncompleteWrite))
        }
    }

    fn receive<'a>(&self, buffer: &'a mut [u8]) -> Result<&'a [u8], Error> {
        let n = self.read(buffer)?;
        log::trace!("Received {} bytes: {:x?}", n, &buffer[..n]);
        if n == buffer.len() {
            Ok(&buffer[1..n])
        } else {
            Ok(&buffer[..n])
        }
    }
}
