// Copyright (C) 2021 Robin Krahl <robin.krahl@ireas.org>
// SPDX-License-Identifier: CC0-1.0

//! Executes vendor-specific command to query the firmware version and the UUID of a Nitrokey 3
//! device.

use std::convert::TryInto as _;

use ctaphid::command::VendorCommand;

const VID_NITROKEY_3: u16 = 0x20a0;
const PID_NITROKEY_3: u16 = 0x42b2;
const COMMAND_VERSION: VendorCommand = VendorCommand::H61;
// const COMMAND_UUID: VendorCommand = VendorCommand::H62;

fn main() -> Result<(), ctaphid::error::Error> {
    env_logger::init();

    let devices = ctaphid::list()?;
    for device in devices
        .iter()
        .filter(|device| device.vendor_id() == VID_NITROKEY_3)
        .filter(|device| device.product_id() == PID_NITROKEY_3)
    {
        let device = devices.connect(device)?;

        let version = device.vendor_command(COMMAND_VERSION, &[])?;
        let version = u32::from_be_bytes(version.try_into().expect("Missing response data"));
        // version = (major << 22) | (minor << 6) | patch
        let major = version >> 22;
        let minor = (version >> 6) & 0b1111_1111_1111_1111;
        let patch = version & 0b11_1111;

        // Currently not supported over CTAPHID
        // let uuid = device.vendor_command(COMMAND_UUID, &[])?;
        // let uuid = u128::from_be_bytes(uuid.try_into().expect("Missing response data"));
        let uuid = 0u128;

        println!("{:X}\tv{}.{}.{}", uuid, major, minor, patch);
    }
    Ok(())
}
