//! Provides a source of data from a vector, used in testing
use crate::{Headers, RowResult, RowStream};

#[derive(Debug, PartialEq)]
pub enum BuildError {
    EmptyIterator,
    FailedHeader,
}

/// A simple struct that helps create RowStreams from vectors.
#[derive(Debug)]
pub struct MockStream<I> {
    iter: I,
    headers: Headers,
}

impl<I> MockStream<I>
where
    I: Iterator<Item = RowResult>,
{
    /// Creates a new object implementing the RowStream trait from an iterator
    /// and some headers.
    pub fn new(iter: I, headers: Headers) -> MockStream<I> {
        MockStream {
            iter,
            headers,
        }
    }

    /// Creates a new object implementing the RowStream trait from an iterator
    /// of RowResult objects, the first of which must be the header row.
    pub fn from_rows(mut iter: I) -> Result<MockStream<I>, BuildError> {
        match iter.next() {
            Some(Ok(row)) => Ok(MockStream::new(iter, row.into())),
            Some(Err(_)) => Err(BuildError::FailedHeader),
            None => Err(BuildError::EmptyIterator),
        }
    }
}

pub struct IntoIter<I> {
    iter: I,
}

impl<I> Iterator for IntoIter<I>
where
    I: Iterator<Item = RowResult>,
{
    type Item = RowResult;

    fn next(&mut self) -> Option<Self::Item> {
        self.iter.next()
    }
}

impl<I> IntoIterator for MockStream<I>
where
    I: Iterator<Item = RowResult>,
{
    type Item = RowResult;

    type IntoIter = IntoIter<I>;

    fn into_iter(self) -> Self::IntoIter {
        IntoIter { iter: self.iter }
    }
}

impl<I> RowStream for MockStream<I>
where
    MockStream<I>: IntoIterator<Item = RowResult>,
{
    fn headers(&self) -> &Headers {
        &self.headers
    }
}

#[cfg(test)]
mod tests {
    use crate::Row;
    use super::{MockStream, RowStream};

    #[test]
    fn test_mock_stream() {
        let m = MockStream::from_rows(
            vec![
                Ok(Row::from(vec!["id", "num"])),
                Ok(Row::from(vec!["1", "40"])),
                Ok(Row::from(vec!["2", "39"])),
            ]
            .into_iter(),
        )
        .unwrap();

        assert_eq!(
            *m.headers(),
            Row::from(vec!["id", "num"]).into()
        );

        let mut m = m.into_iter();

        assert_eq!(m.next().unwrap().unwrap(), Row::from(vec!["1", "40"]));
        assert_eq!(m.next().unwrap().unwrap(), Row::from(vec!["2", "39"]));
    }
}
