use std::fmt;

use crate::{
    margin, padding, position, size,
    unit::{Length, Percent},
};

/// `calc` css function
///
/// Note: I think current implementation doesn't cover every use case, but good
/// enough for most simple cases.
#[derive(Debug, PartialOrd, PartialEq, Clone)]
pub enum Calc {
    // posible values
    Length(Box<Length>),
    Percent(Percent),
    CssVariable(String),

    // posible oprations, these op can be used as value too
    Sum(Box<Calc>, Box<Calc>),
    Sub(Box<Calc>, Box<Calc>),
    Mul(Box<Calc>, f32),
    Div(Box<Calc>, f32),
}

impl From<Length> for Calc {
    fn from(source: Length) -> Self {
        Calc::Length(Box::new(source))
    }
}

impl TryFrom<padding::Length> for Calc {
    type Error = &'static str;

    fn try_from(value: padding::Length) -> Result<Self, Self::Error> {
        match value {
            padding::Length::Length(len) => Ok(len.into()),
            padding::Length::Percent(per) => Ok(per.into()),
            padding::Length::Inherit => Err("Calc cannot accept inherit as value"),
        }
    }
}

impl TryFrom<margin::Length> for Calc {
    type Error = &'static str;

    fn try_from(value: margin::Length) -> Result<Self, Self::Error> {
        match value {
            margin::Length::Length(len) => Ok(len.into()),
            margin::Length::Percent(per) => Ok(per.into()),
            _ => Err("Calc cannot accept values such as inherit, auto ..etc"),
        }
    }
}

impl TryFrom<size::Length> for Calc {
    type Error = &'static str;

    fn try_from(value: size::Length) -> Result<Self, Self::Error> {
        match value {
            size::Length::Length(len) => Ok(len.into()),
            size::Length::Percent(per) => Ok(per.into()),
            _ => Err("Calc cannot accept values such as inherit, auto ..etc"),
        }
    }
}

impl TryFrom<position::PostionLength> for Calc {
    type Error = &'static str;

    fn try_from(value: position::PostionLength) -> Result<Self, Self::Error> {
        match value {
            position::PostionLength::Length(len) => Ok(len.into()),
            position::PostionLength::Percent(per) => Ok(per.into()),
            _ => Err("Calc cannot accept values such as inherit, auto ..etc"),
        }
    }
}

impl From<Percent> for Calc {
    fn from(source: Percent) -> Self {
        Calc::Percent(source)
    }
}

impl From<String> for Calc {
    fn from(source: String) -> Self {
        Calc::CssVariable(source)
    }
}

impl From<&str> for Calc {
    fn from(source: &str) -> Self {
        Calc::CssVariable(source.to_string())
    }
}

impl fmt::Display for Calc {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        fn fmt_calc(calc: &Calc) -> String {
            match calc {
                Calc::Length(val) => val.to_string(),
                Calc::Percent(val) => val.to_string(),
                Calc::CssVariable(val) => format!("var({})", val.to_string()),
                Calc::Sum(c, val) => format!("({} + {})", fmt_calc(c), fmt_calc(val)),
                Calc::Sub(c, val) => format!("({} - {})", fmt_calc(c), fmt_calc(val)),
                Calc::Mul(c, val) => format!("({} * {})", fmt_calc(c), val),
                Calc::Div(c, val) => format!("({} / {})", fmt_calc(c), val),
            }
        }
        match self {
            Self::Length(val) => write!(f, "{}", val),
            Self::Percent(val) => write!(f, "{}", val),
            Self::CssVariable(val) => write!(f, "calc(var({}))", val),
            _ => write!(f, "calc{}", fmt_calc(self)),
        }
    }
}

impl Calc {
    pub fn sum(self, val: impl Into<Calc>) -> Self {
        Self::Sum(Box::new(self), Box::new(val.into()))
    }
    pub fn sub(self, val: impl Into<Calc>) -> Self {
        Self::Sub(Box::new(self), Box::new(val.into()))
    }
    pub fn mul(self, val: f32) -> Self {
        Self::Mul(Box::new(self), val)
    }
    pub fn div(self, val: f32) -> Self {
        Self::Div(Box::new(self), val)
    }
    pub fn try_sum(self, val: Option<impl Into<Calc>>) -> Self {
        if let Some(val) = val {
            self.sum(val)
        } else {
            self
        }
    }
    pub fn try_sub(self, val: Option<impl Into<Calc>>) -> Self {
        if let Some(val) = val {
            self.sub(val)
        } else {
            self
        }
    }
    pub fn try_mul(self, val: Option<f32>) -> Self {
        if let Some(val) = val {
            self.mul(val)
        } else {
            self
        }
    }
    pub fn try_div(self, val: Option<f32>) -> Self {
        if let Some(val) = val {
            self.div(val)
        } else {
            self
        }
    }
}

#[derive(Clone, Debug, PartialEq, Display, From)]
pub enum CalcArgument {
    Length(Length),
    Percent(Percent),
    CssVariable(String),
}

impl From<&str> for CalcArgument {
    fn from(source: &str) -> Self {
        CalcArgument::CssVariable(source.to_string())
    }
}

pub fn calc(value: impl Into<CalcArgument>) -> Calc {
    match value.into() {
        CalcArgument::Length(len) => Calc::Length(Box::new(len)),
        CalcArgument::Percent(per) => Calc::Percent(per),
        CalcArgument::CssVariable(var) => Calc::CssVariable(var),
    }
}
