use chrono::{DateTime, Utc};
use serde::{Deserialize, Serialize, Serializer};
use url::Url;

use crate::definitions::{AcknowledgmentsT, NotesT, ProductGroupsT, ProductsT, ReferencesT};

/// [Vulnerabilities](https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#323-vulnerabilities-property)
#[serde_with::skip_serializing_none]
#[derive(Serialize, Deserialize, Debug)]
pub struct Vulnerability {
    pub acknowledgments: Option<AcknowledgmentsT>,
    pub cve: Option<String>,
    pub cwe: Option<Cwe>,
    pub discovery_date: Option<DateTime<Utc>>,
    pub id: Option<VulnerabilityId>,
    pub involvements: Option<Vec<Involvement>>,
    pub notes: Option<NotesT>,
    pub product_status: Option<ProductStatus>,
    pub references: Option<ReferencesT>,
    pub release_date: Option<DateTime<Utc>>,
    pub remediations: Option<Vec<Remediation>>,
    pub scores: Option<Vec<Score>>,
    pub threats: Option<Vec<Threat>>,
    pub title: Option<String>,
}

/// [CWE](https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#3233-vulnerabilities-property---cwe)
#[derive(Serialize, Deserialize, Debug)]
pub struct Cwe {
    // TODO: Validation - The Weakness ID (id) has value type string with pattern (regular expression):
    pub id: String,
    // TODO: Should be a static map from ID to name for CWE's, ideally sourced from upstream
    pub name: String,
}

/// [Vulnerability ID](https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#3235-vulnerabilities-property---id)
#[derive(Serialize, Deserialize, Debug)]
pub struct VulnerabilityId {
    pub system_name: String,
    pub text: String,
}

/// [Involvements](https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#3236-vulnerabilities-property---involvements)
#[serde_with::skip_serializing_none]
#[derive(Serialize, Deserialize, Debug)]
pub struct Involvement {
    pub party: InvolvementParty,
    pub status: InvolvementStatus,
    pub date: Option<DateTime<Utc>>,
    pub summary: Option<String>,
}

/// [Involvements](https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#3236-vulnerabilities-property---involvements)
#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "lowercase")]
pub enum InvolvementParty {
    Coordinator,
    Discoverer,
    Other,
    User,
    Vendor,
}

/// [Involvements](https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#3236-vulnerabilities-property---involvements)
#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "snake_case")]
pub enum InvolvementStatus {
    Completed,
    ContactAttempted,
    Disputed,
    InProgress,
    NotContacted,
    Open,
}

/// [Product Status](https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#3238-vulnerabilities-property---product-status)
#[serde_with::skip_serializing_none]
#[derive(Serialize, Deserialize, Debug)]
pub struct ProductStatus {
    pub first_affected: Option<ProductsT>,
    pub first_fixed: Option<ProductsT>,
    pub fixed: Option<ProductsT>,
    pub known_affected: Option<ProductsT>,
    pub known_not_affected: Option<ProductsT>,
    pub last_affected: Option<ProductsT>,
    pub recommended: Option<ProductsT>,
    pub under_investigation: Option<ProductsT>,
}

/// [Remediations](https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#32311-vulnerabilities-property---remediations)
#[serde_with::skip_serializing_none]
#[derive(Serialize, Deserialize, Debug)]
pub struct Remediation {
    pub category: RemediationCategory,
    pub details: String,
    pub date: Option<DateTime<Utc>>,
    pub entitlements: Option<Vec<String>>,
    pub group_ids: Option<ProductGroupsT>,
    pub product_ids: Option<ProductsT>,
    pub restart_required: Option<RestartRequired>,
    pub url: Option<Url>,
}

/// [Remediation Category](https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#323111-vulnerabilities-property---remediations---category)
#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "snake_case")]
pub enum RemediationCategory {
    Mitigation,
    NoFixPlanned,
    NoneAvailable,
    VendorFix,
    Workaround,
}

/// [Restart Required](https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#323117-vulnerabilities-property---remediations---restart-required)
#[serde_with::skip_serializing_none]
#[derive(Serialize, Deserialize, Debug)]
pub struct RestartRequired {
    pub category: RestartCategory,
    pub details: Option<String>,
}

/// [Restart Required](https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#323117-vulnerabilities-property---remediations---restart-required)
#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "snake_case")]
pub enum RestartCategory {
    Connected,
    Dependencies,
    Machine,
    None,
    Parent,
    Service,
    System,
    VulnerableComponent,
    Zone,
}

/// [Score](https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#32312-vulnerabilities-property---scores)
#[serde_with::skip_serializing_none]
#[derive(Serialize, Deserialize, Debug)]
pub struct Score {
    pub products: ProductsT,
    // TODO: A CVSS v2 representation
    // TODO: Should have at least one of:
    pub cvss_v2: Option<String>,
    pub cvss_v3: Option<Cvss3>,
}

/// CVSSv3 Representation
///
/// An internal representation of a CVSS score, meant to be serializable to JSON as specified in
/// [https://www.first.org/cvss/cvss-v3.1.json](https://www.first.org/cvss/cvss-v3.1.json).
///
/// Use with [cvss::v3::Base] and the provided `From` implementation.
#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct Cvss3 {
    // TODO: Should be able to store just the Base and generate the rest of this at serialize time?
    version: Cvss3Version,
    vector_string: cvss::v3::Base,
    base_score: f64,
    #[serde(serialize_with = "severity_to_upper")]
    base_severity: cvss::Severity,
}

// https://www.first.org/cvss/cvss-v3.1.json requires baseSeverity field to be uppercase.
// cvss crate normalizes to lowercase.
fn severity_to_upper<S>(severity: &cvss::Severity, serializer: S) -> Result<S::Ok, S::Error>
where
    S: Serializer,
{
    serializer.serialize_str(&severity.as_str().to_uppercase())
}

impl From<cvss::v3::Base> for Cvss3 {
    fn from(b: cvss::v3::Base) -> Self {
        Self {
            version: Cvss3Version::ThreeDotOne,
            vector_string: b.clone(),
            base_score: b.score().value(),
            base_severity: b.severity(),
        }
    }
}

#[derive(Serialize, Deserialize, Debug)]
enum Cvss3Version {
    #[serde(rename = "3.1")]
    ThreeDotOne,
    #[serde(rename = "3.0")]
    ThreeDotZero,
}

/// [Threats](https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#32313-vulnerabilities-property---threats)
#[serde_with::skip_serializing_none]
#[derive(Serialize, Deserialize, Debug)]
pub struct Threat {
    pub category: ThreatCategory,
    pub details: String,
    pub date: Option<DateTime<Utc>>,
    pub group_ids: Option<ProductGroupsT>,
    pub product_ids: Option<ProductsT>,
}

/// [Threats](https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#32313-vulnerabilities-property---threats)
#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "snake_case")]
pub enum ThreatCategory {
    ExploitStatus,
    Impact,
    TargetSet,
}
