use chrono::{DateTime, Utc};
use serde::{Deserialize, Serialize};
use url::Url;

use crate::definitions::{AcknowledgmentsT, NotesT, ProductGroupsT, ProductsT, ReferencesT};

// https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#323-vulnerabilities-property
#[derive(Serialize, Deserialize, Debug)]
pub struct Vulnerability {
    pub acknowledgments: Option<AcknowledgmentsT>,
    // Todo: Constraint/validation
    pub cve: Option<String>,
    pub cwe: Option<Cwe>,
    pub discovery_date: Option<DateTime<Utc>>,
    pub id: Option<VulnerabilityId>,
    pub involvements: Option<Vec<Involvement>>,
    pub notes: Option<NotesT>,
    pub product_status: Option<ProductStatus>,
    pub references: Option<ReferencesT>,
    pub release_date: Option<DateTime<Utc>>,
    pub remediations: Option<Vec<Remediation>>,
    pub scores: Option<Vec<Score>>,
    pub threats: Option<Vec<Threat>>,
    pub title: Option<String>,
}

#[derive(Serialize, Deserialize, Debug)]
pub struct Cwe {
    pub id: String,
    pub name: String,
}

// https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#3235-vulnerabilities-property---id
#[derive(Serialize, Deserialize, Debug)]
pub struct VulnerabilityId {
    pub system_name: String,
    pub text: String,
}

#[derive(Serialize, Deserialize, Debug)]
pub struct Involvement {
    pub party: InvolvementParty,
    pub status: InvolvementStatus,
    pub date: Option<DateTime<Utc>>,
    pub summary: Option<String>,
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "lowercase")]
pub enum InvolvementParty {
    Coordinator,
    Discoverer,
    Other,
    User,
    Vendor,
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "snake_case")]
pub enum InvolvementStatus {
    Completed,
    ContactAttempted,
    Disputed,
    InProgress,
    NotContacted,
    Open,
}

// https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#3238-vulnerabilities-property---product-status
#[derive(Serialize, Deserialize, Debug)]
pub struct ProductStatus {
    pub first_affected: Option<ProductsT>,
    pub first_fixed: Option<ProductsT>,
    pub fixed: Option<ProductsT>,
    pub known_affected: Option<ProductsT>,
    pub known_not_affected: Option<ProductsT>,
    pub last_affected: Option<ProductsT>,
    pub recommended: Option<ProductsT>,
    pub under_investigation: Option<ProductsT>,
}

// https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#32311-vulnerabilities-property---remediations
#[derive(Serialize, Deserialize, Debug)]
pub struct Remediation {
    pub category: RemediationCategory,
    pub details: String,
    pub date: Option<DateTime<Utc>>,
    pub entitlements: Option<Vec<String>>,
    pub group_ids: Option<ProductGroupsT>,
    pub product_ids: Option<ProductsT>,
    pub restart_required: Option<RestartRequired>,
    pub url: Url,
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "snake_case")]
pub enum RemediationCategory {
    Mitigation,
    NoFixPlanned,
    NoneAvailable,
    VendorFix,
    Workaround,
}

#[derive(Serialize, Deserialize, Debug)]
pub struct RestartRequired {
    pub category: RestartCategory,
    pub details: Option<String>,
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "snake_case")]
pub enum RestartCategory {
    Connected,
    Dependencies,
    Machine,
    None,
    Parent,
    Service,
    System,
    VulnerableComponent,
    Zone,
}

// https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#32312-vulnerabilities-property---scores
#[derive(Serialize, Deserialize, Debug)]
pub struct Score {
    pub products: ProductsT,
    // TODO: Should have at least one of:
    pub cvss_v2: Option<String>,
    pub cvss_v3: Option<cvss::v3::Base>,
}

// https://github.com/oasis-tcs/csaf/blob/master/csaf_2.0/prose/csaf-v2-editor-draft.md#32313-vulnerabilities-property---threats
#[derive(Serialize, Deserialize, Debug)]
pub struct Threat {
    pub category: ThreatCategory,
    pub details: String,
    pub date: Option<DateTime<Utc>>,
    pub group_ids: Option<ProductGroupsT>,
    pub product_ids: Option<ProductsT>,
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "snake_case")]
pub enum ThreatCategory {
    ExploitStatus,
    Impact,
    TargetSet,
}
