use rand::distributions::uniform::SampleRange;

use crate::{traits::Random, random_number};

/// Generates a vector of items that implement the `Random` trait.
/// 
/// ### Examples
/// 
/// ```
/// // Note: `test` or `all` crate features must be enabled to access the `test` module.
/// #[cfg(any(feature = "test", feature = "all"))]
/// use cs_utils::test::random_vec;
/// // Note: `traits` or `all` crate features must be enabled to access the `traits` module.
/// #[cfg(any(feature = "traits", feature = "all"))]
/// use cs_utils::traits::Random;
/// use cs_utils::{random_number, random_str};
/// 
/// #[derive(Debug)]
/// struct SomeStruct {
///     id: u16,
///     name: String,
/// }
/// 
/// impl Random for SomeStruct {
///     fn random() -> Self {
///         return SomeStruct {
///             id: random_number(0..=u16::MAX),
///             name: random_str(10),
///         };
///     }
/// }
/// 
/// let items_count = random_number(3..=5);
/// let result: Vec<SomeStruct> = random_vec(items_count);
/// 
/// assert_eq!(
///     result.len() as u32,
///     items_count,
///     "Must create {} items.", &items_count,
/// );
/// println!("{:?}", &result);
/// ```
pub fn random_vec<T: Random>(items_count: u32) -> Vec<T> {
    let mut result = vec![];
    let mut i = items_count;

    while i > 0 {
        result.push(T::random());
        i -= 1;
    }

    return result;
}

/// Generates a vector of items that implement the `Random` trait.
/// 
/// ### Examples
/// 
/// ```
/// use cs_utils::test::random_vec_rg;
/// use cs_utils::traits::Random;
/// use cs_utils::{random_number, random_str};
/// 
/// #[derive(Debug)]
/// struct SomeStruct {
///     id: u16,
///     name: String,
/// }
/// 
/// impl Random for SomeStruct {
///     fn random() -> Self {
///         return SomeStruct {
///             id: random_number(0..=u16::MAX),
///             name: random_str(10),
///         };
///     }
/// }
/// 
/// let result: Vec<SomeStruct> = random_vec_rg(3..7);
/// 
/// assert!(
///     result.len() >= 3,
///     "Must create at least 3 items.",
/// );
/// 
/// assert!(
///     result.len() < 7,
///     "Must create at most 6 items.",
/// );
/// 
/// println!("{:?}", &result);
/// ```
pub fn random_vec_rg<T: Random, R: SampleRange<u32>>(range: R) -> Vec<T> {
    let messages_count = random_number(range);

    return random_vec::<T>(messages_count);
}
