use std::io::{Error, ErrorKind};

use bytes::BytesMut;
use serde::{Serialize, de::DeserializeOwned};
use tokio_util::codec::Decoder;

use super::GenericLinesCodec;

impl<T: Serialize + DeserializeOwned> Decoder for GenericLinesCodec<T> {
    type Item = T;
    type Error = Error;

    fn decode(
        &mut self,
        buf: &mut BytesMut,
    ) -> Result<Option<Self::Item>, Self::Error> {
        let str_result = Decoder::decode(
            &mut self.length_delimited_codec, 
            buf,
        );

        let maybe_str = match str_result {
            Ok(s) => s,
            Err(err) => return Err(
                Error::new(
                    ErrorKind::InvalidData,
                    format!("Cannot decode input: {:?}.", err),
                ),
            ),
        };

        let string = match maybe_str {
            None => return Ok(None),
            Some(b) => b,
        };

        let message: T = serde_json::from_str(&string)
            .expect("Cannot deserialize bytes to message.");

        return Ok(Some(message));
    } 
}
