// use std::fmt::{self, Write};

// /// Formatter aims to help to write nice log output.
// /// 
// /// ### Examples
// /// 
// /// ```
// /// ```
// pub struct LogFormatter<'a, 'b> {
//     formatter: &'a mut fmt::Formatter<'b>,
//     total_width: usize,
//     ident: String,
// }

// // formatter.write_fmt(format_args!("\nDNS record:"))?;

// // // TODO: use one from utils-rs
// // let mut packet_formatter = NetworkPacketFormatter::new(
// //     formatter,
// //     50,
// //     "  ",
// // );

// // packet_formatter.write("Hostname:", &self.hostname())?;
// // packet_formatter.write("IP Address:", &self.ip_address())?;
// // packet_formatter.write("TTL:", &self.ttl())?;

// // return Ok(());

// impl<'a, 'b> LogFormatter<'a, 'b> {
//     pub fn new<StrRef: AsRef<str>>(
//         formatter: &'a mut fmt::Formatter<'b>,
//         total_width: usize,
//         ident: StrRef,
//     ) -> Self {
//         let ident = ident.as_ref().to_string();

//         return LogFormatter {
//             formatter,
//             total_width,
//             ident,
//         };
//     }

//     pub fn write_line<S: AsRef<str>, T: AsRef<str>>(
//         &mut self,
//         caption: S,
//         data: T,
//     ) -> Result<(), fmt::Error> {
//         let data_str = data.as_ref();
//         let caption_str = caption.as_ref();
//         let width = self.total_width - self.ident.len() - caption_str.len();
    
//         write!(
//             self.formatter,
//             "\n{ident}{caption}{:.>width$}",
//             data_str,
//             width = width,
//             ident = &self.ident,
//             caption = caption_str,
//         )?;
    
//         return Ok(());
//     }

//     pub fn write<S: AsRef<str>>(
//         &mut self,
//         line: S,
//     ) -> Result<(), fmt::Error> {

//         let line = line.as_ref();
//         write!(
//             self.formatter,
//             "\n{ident}{line}",
//             line = line,
//             ident = &self.ident,
//         )?;
    
//         return Ok(());
//     }

//     pub fn caption<S: AsRef<str>>(
//         &mut self,
//         caption: S,
//     ) -> Result<(), fmt::Error> {
//         self.formatter
//             .write_str(&LogFormatter::make_caption(caption))?;

//         return Ok(());
//     }

//     pub fn make_caption<S: AsRef<str>>(
//         caption: S,
//     ) -> String {
//         return format!("\n## {}:", &caption.as_ref());
//     }

//     pub fn print_caption<S: AsRef<str>>(
//         caption: S,
//     ) {
//         return println!("{}", &LogFormatter::make_caption(caption));
//     }
// }


/// Formatter aims to help to write nice log output.
/// 
/// ### Examples
/// 
/// ```
/// ```
pub struct LogFormatter {
    total_width: usize,
    ident: String,
}

use std::{fmt::{Display, self}, io::{self, ErrorKind}, str, marker::PhantomData};

pub struct FmtWriteToIoWrite<'a, T: fmt::Write + 'a> {
    fmt_writer: T,
    _ph: &'a PhantomData<T>,
}

impl<'a, T: fmt::Write + 'a> io::Write for FmtWriteToIoWrite<'a, T> {
    fn write(&mut self, buf: &[u8]) -> io::Result<usize> {
        let string = str::from_utf8(&buf[..])
            .expect("Cannot convert buffer to UTF8 string.");
        
        match self.fmt_writer.write_str(string) {
            Ok(_) => {
                return Ok(buf.len());
            },
            Err(error) => {
                let io_error = io::Error::new(
                    ErrorKind::WriteZero,
                    error.to_string(),
                );
                return Err(io_error);
            },
        }
    }

    fn flush(&mut self) -> io::Result<()> {
        return Ok(());
    }
}

impl<'a, T: fmt::Write + 'a> FmtWriteToIoWrite<'a, T> {
    pub fn new(
        fmt_writer: &'a mut T,
    ) -> Box<dyn io::Write + 'a> {
        return Box::new(
            FmtWriteToIoWrite {
                fmt_writer,
                _ph: &PhantomData,
            },
        );
    }
}

impl LogFormatter {
    pub fn new<StrRef: AsRef<str>>(
        total_width: usize,
        ident: StrRef,
    ) -> Self {
        let ident = ident.as_ref().to_string();

        // TODO: test this
        assert!(
            ident.len() < total_width,
            "Identation length must be smaller than total width.",
        );

        return LogFormatter {
            total_width,
            ident,
        };
    }

    pub fn caption<S: AsRef<str>>(
        &self,
        caption: S,
    ) -> String {
        let ident_size = self.ident.len() / 2;

        return format!(
            "\n{ident}{: >width$}{caption}\n\n",
            ident = "",
            caption = caption.as_ref(),
            width = ident_size,
        );
    }

    pub fn value_line<S: AsRef<str>, T: Display>(
        &self,
        label: S,
        value: T,
    ) -> String {
        let value_str = label.as_ref();
        let width_substract = self.ident.len() - value_str.len();

        // TODO: test this
        let width = if self.total_width >= width_substract {
            self.total_width - width_substract
        } else {
            self.total_width
        };

        return format!(
            "{ident}{value}{:.>width$}\n",
            value,
            ident = &self.ident,
            value = value_str,
            width = width,
        );
    }

    pub fn line<S: AsRef<str>>(
        &self,
        line: S,
    ) -> String {
        let line = line.as_ref();

        return format!(
            "{ident}{line}",
            line = line,
            ident = &self.ident,
        );
    }

    pub fn line_start<S: AsRef<str>>(
        &self,
        label: S,
    ) -> String {
        let width_substract = self.ident.len() - label.as_ref().len();

        // TODO: test this
        let width = if self.total_width >= width_substract {
            self.total_width - width_substract
        } else {
            self.total_width
        };
    
        return format!(
            "{ident}{caption}{:>width$}\n",
            ident = &self.ident,
            caption = label.as_ref(),
            width = width,
        );
    }

    pub fn line_end<S: AsRef<str>>(
        &self,
        label: S,
    ) -> String {
        let width_substract = self.ident.len();

        // TODO: test this
        let width = if self.total_width >= width_substract {
            self.total_width - width_substract
        } else {
            self.total_width
        };
    
        return format!(
            "{ident}{:>width$}\n",
            label.as_ref(),
            ident = &self.ident,
            width = width,
        );
    }

    pub fn underline(&self) -> String {
        let width = self.total_width;

        return format!(
            "{ident}{:->width$}\n",
            ident = &self.ident,
            width = width,
        );
    }
}

pub struct WriteFormatter<'a> {
    log_formatter: LogFormatter,
    writer: Box<dyn io::Write + 'a>,
}

impl<'a> WriteFormatter<'a> {
    pub fn new<StrRef: AsRef<str>, W: io::Write + 'static>(
        total_width: usize,
        ident: StrRef,
        writer: W,
    ) -> Self {
        let ident = ident.as_ref().to_string();

        return WriteFormatter {
            log_formatter: LogFormatter::new(total_width, ident),
            writer: Box::new(writer),
        };
    }

    pub fn new_from_fmt<StrRef: AsRef<str>, R: fmt::Write + 'a>(
        total_width: usize,
        ident: StrRef,
        formatter: &'a mut R,
    ) -> Self {
        let writer = FmtWriteToIoWrite::new(formatter);

        return WriteFormatter {
            log_formatter: LogFormatter::new(total_width, ident),
            writer,
        };
    }

    pub fn caption<S: AsRef<str>>(
        &mut self,
        caption: S,
    ) -> &Self {
        let message = self.log_formatter.caption(caption);

        self.writer.write_all(message.as_bytes())
            .expect("Failed to write caption.");

        return self;
    }

    pub fn value_line<S: AsRef<str>, T: Display>(
        &mut self,
        label: S,
        value: T,
    ) -> &Self {
        let message = self.log_formatter.value_line(label, value);

        self.writer.write_all(message.as_bytes())
            .expect("Failed to write value line.");

        return self;
    }

    pub fn line<S: AsRef<str>>(
        &mut self,
        line_str: S,
    ) -> &Self {
        let message = self.log_formatter.line(line_str);

        self.writer.write_all(message.as_bytes())
            .expect("Failed to write underline.");

        return self;
    }

    pub fn underline(&mut self) -> &Self {
        let message = self.log_formatter.underline();

        self.writer.write_all(message.as_bytes())
            .expect("Failed to write underline.");

        return self;
    }

    pub fn line_start<S: AsRef<str>>(
        &mut self,
        label: S,
    ) -> &Self {
        let message = self.log_formatter.line_start(label);

        self.writer.write_all(message.as_bytes())
            .expect("Failed to write lien start.");

        return self;
    }

    pub fn line_end<S: AsRef<str>>(
        &mut self,
        label: S,
    ) -> &Self {
        let message = self.log_formatter.line_end(label);

        self.writer.write_all(message.as_bytes())
            .expect("Failed to write line end.");

        return self;
    }
}

#[cfg(test)]
mod tests {
    mod caption {
        use k9::assert_matches_snapshot;

        use crate::LogFormatter;

        #[test]
        fn creates_caption_from_str() {
            let formatter = LogFormatter::new(50, "  ");
            let test_string = "Start".to_string();

            assert_matches_snapshot!(
                formatter.caption(&test_string),
                "Must create formatted caption (str)."
            );

            assert_matches_snapshot!(
                formatter.caption(test_string.as_str()),
                "Must create formatted caption (as_str)."
            );

            assert_matches_snapshot!(
                formatter.caption("Start"),
                "Must create formatted caption (str)."
            );
        }

        #[test]
        fn creates_caption_from_string() {
            let formatter = LogFormatter::new(50, "  ");

            assert_matches_snapshot!(
                formatter.caption("Start".to_string()),
                "Must create formatted caption (string)."
            );
        }
    }

    mod line {
        use k9::assert_matches_snapshot;
        use rstest::rstest;
        use crate::LogFormatter;

        #[rstest]
        #[case(&("some-string".to_string()))]
        #[case("some-string")]
        #[test]
        fn creates_line_for_str<T: Display>(
            #[case] test_string: T,
        ) {
            let formatter = LogFormatter::new(50, "  ");

            assert_matches_snapshot!(
                formatter.value_line("Name", test_string),
                "Must create formatted line (str)."
            );
        }

        #[rstest]
        #[case(&("some-different-string".to_string()))]
        #[case("some-different-string")]
        #[test]
        fn creates_line_for_as_str<T: Display>(
            #[case] test_string: T,
        ) {
            let formatter = LogFormatter::new(50, "  ");
            let test_string = test_string.to_string();

            assert_matches_snapshot!(
                formatter.value_line("Name", test_string.as_str()),
                "Must create formatted line (as_str)."
            );
        }

        #[test]
        fn creates_line_for_string() {
            let formatter = LogFormatter::new(50, "  ");

            assert_matches_snapshot!(
                formatter.value_line("Name", "value".to_string()),
                "Must create formatted line (string)."
            );
        }

        use std::fmt::Display;

        struct TestStruct {}

        impl Display for TestStruct {
            fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
                f.write_fmt(
                    format_args!("display output: {}", "200"),
                )?;

                return Ok(());
            }
        }

        #[rstest]
        #[case(500)]
        #[case(true)]
        #[case(false)]
        #[case(TestStruct {})]
        #[test]
        fn creates_line_for_display_items<T: Display>(
            #[case] data: T,
        ) {
            let formatter = LogFormatter::new(50, "  ");

            assert_matches_snapshot!(
                formatter.value_line("Name", data),
                "Must create formatted line (display)."
            );
        }
    }

    mod ident {
        use k9::assert_matches_snapshot;
        use rstest::rstest;
        use crate::LogFormatter;

        #[rstest]
        #[case(&("some-other-string".to_string()))]
        #[case("some-other-string")]
        #[test]
        fn idents_str<S: AsRef<str>>(
            #[case] test_string: S,
        ) {
            let formatter = LogFormatter::new(50, "  ");

            assert_matches_snapshot!(
                formatter.line(test_string),
                "Must ident a line (str)."
            );
        }

        #[rstest]
        #[case(&("some-other-string".to_string()))]
        #[case("some-other-string")]
        #[test]
        fn idents_as_str<S: AsRef<str>>(
            #[case] test_string: S,
        ) {
            let formatter = LogFormatter::new(50, "  ");
            let test_string = test_string.as_ref().to_string();

            assert_matches_snapshot!(
                formatter.line(test_string.as_str()),
                "Must ident a line (as_str)."
            );
        }

        #[test]
        fn idents_string() {
            let formatter = LogFormatter::new(50, "  ");
            let test_string = "string-for-identation".to_string();

            assert_matches_snapshot!(
                formatter.line(test_string),
                "Must ident a line (string)."
            );
        }
    }
}
