use rand::{Rng, distributions::{Alphanumeric, uniform::SampleRange}};

use crate::random_number;

/// Generate a random string, `length` characters long.
/// 
/// ## Panics
/// 
/// Panics if length of `0` passed, as its impossible to have a zero-length `random` string.
/// 
/// ## Arguments
/// 
/// * `length` - resulting random string length, must be greater than 0.
/// 
/// ## Examples
/// 
/// ```
/// use cs_utils::random_str;
/// 
/// let result = random_str(8);
/// 
/// println!("random string \"{}\" generated", &result);
/// 
/// assert_eq!(
///     result.len(),
///     8,
/// );
/// ```
pub fn random_str(length: usize) -> String {
    assert!(
        length > 0,
        "Length parameter should be greater than 0.",
    );

    return rand::thread_rng()
        .sample_iter(&Alphanumeric)
        .take(length)
        .map(char::from)
        .collect();
}


/// Generate a random string of `random` length.
/// 
/// ## Panics
/// 
/// Panics if length of `0` generated by the range, as its impossible to have a zero-length `random` string.
/// 
/// ## Arguments
/// 
/// * `range` - range(min, max]) of the string length.
/// 
/// ## Examples
/// 
/// ```
/// use cs_utils::random_str_rg;
/// 
/// let result = random_str_rg(2..8);
/// 
/// println!("random string \"{}\" generated", &result);
/// 
/// assert!(
///     result.len() >= 2,
///     "Generated string must be at least 2 characters long."
/// );
/// 
/// assert!(
///     result.len() < 8,
///     "Generated string length must be < 7.",
/// );
/// ```
pub fn random_str_rg<R: SampleRange<usize>>(range: R) -> String {
    let string_length = random_number(range);

    return random_str(string_length);
}
