use rand::distributions::uniform::SampleRange;
use tokio::time;

use crate::random_number;

/// Asynchronously wait for `ms` milliseconds.
/// 
/// ### Examples
/// 
/// ```
/// use cs_utils::futures::wait;
/// use tokio::time::{Instant, Duration};
/// 
/// #[tokio::main]
/// async fn main() {
///     let start_time = Instant::now();
///     println!("start");
///     wait(1000).await;
///     println!("end");
///     
///     assert!(
///         (Instant::now() - start_time) >= Duration::from_millis(1000),
///     );
/// }
/// 
/// ```
pub async fn wait(ms: u64) {
    time::sleep(time::Duration::from_millis(ms)).await;
}

/// Asynchronously wait for some `random number` of milliseconds.
/// 
/// ### Examples
/// 
/// ```
/// use cs_utils::futures::wait_random;
/// use tokio::time::{Instant, Duration};
/// 
/// #[tokio::main]
/// async fn main() {
///     let start_time = Instant::now();
///     println!("start");
///     wait_random(500..1000).await;
///     println!("end");
///     
///     assert!(
///         (Instant::now() - start_time) >= Duration::from_millis(500),
///     );
/// 
///     assert!(
///         (Instant::now() - start_time) <= Duration::from_millis(1000 + 100),
///     );
/// }
/// 
/// ```
pub async fn wait_random<T: SampleRange<u64>>(range: T) {
    let ms = random_number(range);

    time::sleep(time::Duration::from_millis(ms)).await;
}
