use tracing::{info, trace, debug, warn, error, Span};
use crate::Tracer;

pub struct Trace {
    trace_tree_name: &'static str,
    is_silent: bool,
    trace_span: Span,
    info_span: Span,
    debug_span: Span,
    warn_span: Span,
    error_span: Span,
}

impl std::fmt::Debug for Trace {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        return f.debug_struct("Trace")
            .field("name", &self.trace_tree_name)
            .finish();
    }
}

impl Tracer for Trace {
    fn trace_tree_name(&self) -> &'static str {
        return &self.trace_tree_name;
    }

    fn is_silent(&self) -> bool {
        return self.is_silent;
    }

    fn as_silent(mut self, is_silent: bool) -> Box<dyn Tracer> {
        self.is_silent = is_silent;

        return Box::new(
            Trace {
                is_silent,
                ..self
            },
        );
    }

    fn span(&self) -> &Span {
        return &self.trace_span;
    }

    // fn child(&self, child_name: &'static str) -> Box<dyn Tracer> {

    //     unsafe { NAME1 = "child_name"; }

    //     let child_trace = Trace {
    //         is_silent: self.is_silent,
    //         trace_span: trace_span!(parent: &self.trace_span, "aaa"),
    //         info_span: info_span!(parent: &self.info_span, unsafe { NAME1 }),
    //         debug_span: debug_span!(parent: &self.debug_span, unsafe { NAME1 }),
    //         warn_span: warn_span!(parent: &self.warn_span, unsafe { NAME1 }),
    //         error_span: error_span!(parent: &self.error_span, unsafe { NAME1 }),
    //         name: child_name.to_string(),
    //     };

    //     return Box::new(
    //         child_trace.as_silent(self.is_silent),
    //     );
    // }

    fn trace(&self, message: &str) -> &dyn Tracer {
        if self.is_silent {
            return self;
        }

        let _enter = self.trace_span.enter();
        trace!(message);
        
        return self;
    }

    fn debug(&self, message: &str) -> &dyn Tracer {
        if self.is_silent {
            return self;
        }

        let _enter = self.debug_span.enter();
        debug!(message);

        return self;
    }

    fn info(&self, message: &str) -> &dyn Tracer {
        if self.is_silent {
            return self;
        }

        let _enter = self.info_span.enter();
        info!(message);

        return self;
    }

    fn warn(&self, message: &str) -> &dyn Tracer {
        if self.is_silent {
            return self;
        }

        let _enter = self.warn_span.enter();
        warn!(message);

        return self;
    }

    fn error(&self, message: &str) -> &dyn Tracer {
        if self.is_silent {
            return self;
        }

        let _enter = self.error_span.enter();
        error!(message);

        return self;
    }
}

impl Trace {
    pub fn new(
        name: &'static str,
        is_silent: bool,
        trace_span: Span,
        info_span: Span,
        debug_span: Span,
        warn_span: Span,
        error_span: Span,
    ) -> Box<dyn Tracer> {
        return Box::new(
            Trace {
                is_silent,
                trace_span,
                info_span,
                debug_span,
                warn_span,
                error_span,
                trace_tree_name: name,
            },
        );
    }
}

#[macro_export]
macro_rules! create_trace {
    ($name:expr) => {
        {
            cs_trace::Trace::new(
                $name,
                false,
                cs_trace::tracing_macros::trace_span!($name, cs_trace = cs_trace::EMPTY_FIELD),
                cs_trace::tracing_macros::info_span!($name, cs_trace = cs_trace::EMPTY_FIELD),
                cs_trace::tracing_macros::debug_span!($name, cs_trace = cs_trace::EMPTY_FIELD),
                cs_trace::tracing_macros::warn_span!($name, cs_trace = cs_trace::EMPTY_FIELD),
                cs_trace::tracing_macros::error_span!($name, cs_trace = cs_trace::EMPTY_FIELD),
            )
        };
    };
}

#[macro_export]
macro_rules! child {
    ($parent_trace:ident, $name:expr) => {
        {
            let parent_span = $parent_trace.span();

            cs_trace::Trace::new(
                $parent_trace.trace_tree_name(),
                $parent_trace.is_silent(),
                cs_trace::tracing_macros::trace_span!(parent: parent_span, $name, cs_trace = cs_trace::EMPTY_FIELD),
                cs_trace::tracing_macros::info_span!(parent: parent_span, $name, cs_trace = cs_trace::EMPTY_FIELD),
                cs_trace::tracing_macros::debug_span!(parent: parent_span, $name, cs_trace = cs_trace::EMPTY_FIELD),
                cs_trace::tracing_macros::warn_span!(parent: parent_span, $name, cs_trace = cs_trace::EMPTY_FIELD),
                cs_trace::tracing_macros::error_span!(parent: parent_span, $name, cs_trace = cs_trace::EMPTY_FIELD),
            )
        };
    };
}
