use std::{io::{self, Write}, sync::{Arc, Mutex}};

use tracing::Metadata;
use tracing_subscriber::fmt::MakeWriter;

pub struct CompositeWriter{
    writers: Arc<Mutex<Vec<Box<dyn Write + Send + Sync>>>>,
}

impl CompositeWriter {
    pub fn new(
        writers: Vec<Box<dyn Write + Send + Sync>>,
    ) -> CompositeWriter {
        return CompositeWriter {
                writers: Arc::new(Mutex::new(writers)),
            };
    }

    pub fn write_impl(&self, buf: &[u8]) -> io::Result<usize> {
        // TODO: make writers independent
        let mut items = self.writers.lock().unwrap();

        for writer in items.iter_mut() {
            let _result = writer.write(&buf);
        }

        return Ok(buf.len());
    }

    pub fn flush_impl(&self) -> io::Result<()> {
        // TODO: make writers independent
        let mut items = self.writers.lock().unwrap();

        for writer in items.iter_mut() {
            let _result = writer.flush();
        }

        return Ok(());
    }
}

pub struct MakeCompositeWriter {
    composite_writer: CompositeWriter,
}

impl MakeCompositeWriter {
    pub fn new(
        writers: Vec<Box<dyn Write + Send + Sync>>,
    ) -> Self {
        return MakeCompositeWriter {
            composite_writer: CompositeWriter::new(writers),
        };
    }
}

impl<'a> MakeWriter<'a> for MakeCompositeWriter {
    type Writer = &'a CompositeWriter;

    fn make_writer(&'a self) -> Self::Writer {
        return &self.composite_writer;
    }

    fn make_writer_for(&'a self, _meta: &Metadata<'_>) -> Self::Writer {
        return &self.composite_writer;
    }
}

impl From<Vec<Box<dyn Write + Send + Sync>>> for CompositeWriter {
    fn from(writers: Vec<Box<dyn Write + Send + Sync>>) -> CompositeWriter {
        return CompositeWriter {
            writers: Arc::new(Mutex::new(writers)),
        };
    }
}

impl From<Vec<Box<dyn Write + Send + Sync>>> for MakeCompositeWriter {
    fn from(writers: Vec<Box<dyn Write + Send + Sync>>) -> MakeCompositeWriter {
        return MakeCompositeWriter::new(writers);
    }
}

// impl<'a> Write for CompositeWriter<'a>
//  {
//     fn write(&mut self, buf: &[u8]) -> io::Result<usize> {
//         return self.write(buf);
//     }

//     fn flush(&mut self) -> io::Result<()> {
//         return self.flush();
//     }
// }

impl<'a> Write for &'a CompositeWriter {
    fn write(&mut self, buf: &[u8]) -> io::Result<usize> {
        return self.write_impl(buf);
    }

    fn flush(&mut self) -> io::Result<()> {
        return self.flush_impl();
    }
}
